/* -*- C -*-
 * 
 * SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
 * for finite-state functional transition systems
 *
 * Main procedure
 *
 * Author: Alberto Griggio <griggio@fbk.eu>
 * See LICENSE.txt for copyright/licensing information
 * See CREDITS.txt for other credits
 */

#include "simplic3.h"
#include "simplic3/miscutils.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>

SimplIC3 *the_prover = NULL;
int print_stats = 0;

void print_prover_stats(SimplIC3 *s)
{
    char **stats;
    char *cur;
    double tot;
    size_t j;
    stats = simplic3_stats(*s);
    cur = stats[0];
    j = 0;
    while (cur != NULL) {
        printf("%s = ", cur);
        simplic3_free(cur);
        ++j;
        cur = stats[j];
        printf("%s\n", cur);
        simplic3_free(cur);
        ++j;
        cur = stats[j];
    }
    simplic3_free(stats);
    
    tot = simplic3::get_cpu_time_sec();
    printf("total_time = %.3f\n", tot);
}


void handle_interrupt(int signum)
{
    if (print_stats) {
        printf("interrupted by signal %d\n", signum);
        if (the_prover) {
            print_prover_stats(the_prover);
        }
    }
    printf("Unknown\n");
    fflush(stdout);
    _exit(signum);
}


int main(int argc, const char **argv)
{
    aiger *aigmgr;
    SimplIC3 s;
    int i, j;
    int res, exitcode;
    const char *filename;
    const char *str;
    const char *wit_filename;
    int prop;
    int witness;

    filename = NULL;
    wit_filename = NULL;
    j = 0;
    exitcode = 2; /* unknown result / error */
    prop = 0;
    witness = 0;
    print_stats = 0;

    aigmgr = aiger_init();
    s = simplic3_new();

    signal(SIGXCPU, handle_interrupt);
    signal(SIGINT, handle_interrupt);

    if (SIMPLIC3_ERROR(s)) {
        goto cleanup;
    }
    
    for (i = 1; i < argc; ++i) {
        if (strcmp(argv[i], "-h") == 0) {
            char *hs = simplic3_opthelp(s);
            fprintf(stdout,
                    "USAGE: simplic3 [OPTIONS] filename.aig\nOPTIONS:\n"
                    "%s  -n N : set index of property to prove (0 by default)\n"
                    "  -w : produce witness (cex or invariant)\n"
                    "  -o NAME : write witness into the given file NAME\n"
                    "  -h : print this help message\n", hs);
            simplic3_free(hs);
            goto cleanup;
        } else if (strcmp(argv[i], "-w") == 0) {
            witness = 1;
        } else if (strcmp(argv[i], "-n") == 0) {
            if (i+1 < argc) {
                prop = atoi(argv[i+1]);
                ++i;
            } else {
                fprintf(stderr, "ERROR, missing property index\n");
                goto cleanup;
            }
        } else if (strcmp(argv[i], "-o") == 0) {
            if (i+1 < argc) {
                wit_filename = argv[i+1];
                ++i;
            } else {
                fprintf(stderr, "ERROR, missing output file name\n");
                goto cleanup;
            }
        } else if (argv[i][0] == '-') {
            const char *opt = argv[i];
            str = "";
            if (i+1 < argc) {
                str = argv[i+1];
                ++i;
            }
            if (simplic3_setopt(s, opt, str) != 0) {
                fprintf(stderr,
                        "ERROR setting option %s %s (use -h for help)\n",
                        opt, str);
                goto cleanup;
            }
        } else {
            filename = argv[i];
            j = i;
        }
    }

    if (filename == NULL) {
        fprintf(stderr, "ERROR, missing filename\n");
        goto cleanup;
    }

    str = aiger_open_and_read_from_file(aigmgr, filename);

    if (str) {
        fprintf(stderr, "ERROR: %s\n", str);
        goto cleanup;
    }

    res = simplic3_init(s, aigmgr);
    if (res != 0) {
        fprintf(stderr, "ERROR initializing SimplIC3\n");
        goto cleanup;
    }

    str = simplic3_getopt(s, "-v");
    print_stats = (str && atoi(str) > 0);
    
    the_prover = &s;
    res = simplic3_prove(s, prop);

    if (res == -1) {
        fprintf(stderr, "ERROR proving property %d\n", prop);
        goto cleanup;
    }

    if (print_stats) {
        print_prover_stats(&s);
        /* char **stats; */
        /* char *cur; */
        /* double tot; */
        /* stats = simplic3_stats(s); */
        /* cur = stats[0]; */
        /* j = 0; */
        /* while (cur != NULL) { */
        /*     printf("%s = ", cur); */
        /*     simplic3_free(cur); */
        /*     ++j; */
        /*     cur = stats[j]; */
        /*     printf("%s\n", cur); */
        /*     simplic3_free(cur); */
        /*     ++j; */
        /*     cur = stats[j]; */
        /* } */
        /* simplic3_free(stats); */

        /* tot = simplic3::get_cpu_time_sec(); */
        /* printf("total_time = %.3f\n", tot); */
    }

    if (witness) {
        unsigned int *wit;
        int done;
        FILE *out = stdout;
        wit = simplic3_witness(s);
        done = 0;

        if (wit) {
            if (wit_filename) {
                out = fopen(wit_filename, "w");
                if (!out) {
                    fprintf(stderr, "ERROR opening output file %s\n",
                            wit_filename);
                    out = stdout;
                }
            }

            if (out == stdout) {
                fprintf(out, "Witness:\n");
            }

            j = 0;
            i = 0;
            while (!done) {
                if (res == 0) {
                    fprintf(out, "[ ");
                } else {
                    fprintf(out, "%d: ", i);
                }
                while (!done) {
                    unsigned lit = wit[j];
                    ++j;
                    if (!lit) {
                        ++i;
                        if (!wit[j]) {
                            done = 1;
                        }
                        if (res == 0) {
                            fprintf(out, "]");
                        }
                        fprintf(out, "\n");
                        break;
                    } else {
                        fprintf(out, "%s%d ", aiger_sign(lit) ? "~" : "",
                                aiger_lit2var(lit));
                    }
                }
            }
            fflush(out);
            if (out != stdout) {
                fclose(out);
            }

            simplic3_free(wit);
        } else {
            fprintf(stderr, "ERROR computing witness\n");
        }
    }

    if (res == 0) {
        exitcode = 0;
        printf("Safe\n");
    } else {
        exitcode = 1;
        printf("Unsafe\n");
    }
    fflush(stdout);

  cleanup:
    simplic3_delete(s);
    aiger_reset(aigmgr);
    return exitcode;
}
