/* -*- C -*-
 * 
 * SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
 * for finite-state functional transition systems
 *
 * C API
 *
 * Author: Alberto Griggio <griggio@fbk.eu>
 * See LICENSE.txt for copyright/licensing information
 * See CREDITS.txt for other credits
 */

#ifndef SIMPLIC3_SIMPLIC3_H
#define SIMPLIC3_SIMPLIC3_H

#ifdef __cplusplus
extern "C" {
#endif

#include "simplic3/aiger.h"

/**
 * Abstract data structure for the Simple IC3 solver
 */
typedef struct SimplIC3 { void *repr; } SimplIC3;

/**
 * error checking macro
 */
#define SIMPLIC3_ERROR(s) ((s).repr == NULL)

/**
 * memory deallocation routine for results allocated by the SimplIC3 API
 */
void simplic3_free(void *mem);

/**
 * creates a new SimplIC3 solver
 */
SimplIC3 simplic3_new(void);

/**
 * destroys a SimplIC3 solver
 *
 * \param s The SimplIC3 solver to destroy
 */
void simplic3_delete(SimplIC3 s);

/**
 * initializes the solver with the given Aiger circuit.
 *
 * Must be called before ::simplic3_prove(), and can be called only once
 *
 * \param s The SimplIC3 solver
 * \param aigmgr The Aiger manager
 *
 * \return 0 on success, nonzero on error
 */
int simplic3_init(SimplIC3 s, aiger *aigmgr);

/**
 * sets a SimplIC3 option
 *
 * \param s The SimplIC3 solver
 * \param opt The option name
 * \param val The option value
 *
 * \return 0 on success, nonzero on error
 */
int simplic3_setopt(SimplIC3 s, const char *opt, const char *val);

/**
 * retrieves the value of a SimplIC3 option
 *
 * \param s The SimplIC3 solver
 * \param opt The option name
 *
 * \return the option value, or NULL on error
 */
const char *simplic3_getopt(SimplIC3 s, const char *opt);

/**
 * retrieves the help message for the SimplIC3 options
 *
 * \param s The SimplIC3 solver
 *
 * \return the help message, or NULL on error. The string must be deallocated
 * by the user with ::simplic3_free().
 */
char *simplic3_opthelp(SimplIC3 s);

/**
 * set a callback function to be notified of safe bounds
 *
 * The callback function is called with input K as soon as the property is
 * determined to hold for all states reachable in at most K steps from the
 * initial states
 *
 * \param s The SimplIC3 solver
 * \param cb The callback function
 * \param user_data a generic pointer to arbitrary user-specific data
 *
 * \return 0 on success, nonzero on error
 */
int simplic3_set_safe_bound_callback(SimplIC3 s, void(*cb)(int, void *),
                                     void *user_data);

/**
 * checks the given invariant property
 *
 * \param s The SimplIC3 solver
 * \param prop_idx The index of the property to check in the Aiger circuit
 *                 (see ::simplic3_init()). Property indices start from 0,
 *                 with aiger outputs preceding aiger badstates. For example
 *                 if the Aiger has 2 outputs and 1 bad, and we want to check
 *                 the bad, the index to use will be 2.
 *
 * \return 0 if the property holds, 1 if it is falsifed, -1 on error.
 */
int simplic3_prove(SimplIC3 s, unsigned int prop_idx);

/**
 * retrieves the witness for the last property checked.
 *
 * A witness is a list of Aiger literals, terminated by two consecutive
 * zeros. For satisfied properties, the witness is an inductive invariant in
 * CNF, where the clauses are separated by zero literals.
 *
 * For falsified properties, the witness is a sequence of assignments to state
 * and input variables, where the different steps are separated by
 * zeros. Values for inputs at step i correspond to inputs for the transition
 * from i to i+1 (i.e. for the first transition the input values are recorded
 * at step 0).
 *
 * The return value must be deallocated by the user with ::simplic3_free().
 *
 * \param s The SimplIC3 solver
 *
 * \return the witness for the last property checked, or NULL on error.
 */
unsigned int *simplic3_witness(SimplIC3 s);

/**
 * retrieves the statistics relative to the the last ::simplic3_prove() call.
 *
 * The return value is an array of strings, where two consecutive elements
 * correspond to a (name, value) pair of stats. The array is terminated by a
 * NULL element. Both the array and the individual elements must be
 * deallocated by the user with ::simplic3_free().
 *
 * \param s The SimplIC3 solver
 *
 * \return the list of stats, or NULL on error.
 */
char **simplic3_stats(SimplIC3 s);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* SIMPLIC3_SIMPLIC3_H */
