// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// And-inverter graphs with local 2-level rewriting
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "aig.h"
#include "aigsopdata.h"
#include <algorithm>
#include <assert.h>
#include <sstream>

namespace simplic3 {

AigManager::AigManager():
    next_id_(1)
{
    Aig_ t(0);
    aig_t_ = mk(insert(&t), false);
    aig_f_ = aig_not(aig_t_);

    sopdata_ = NULL;
}


AigManager::~AigManager()
{
    if (sopdata_) {
        delete sopdata_;
    }
}


Aig AigManager::aig_true()
{
    return aig_t_;
}
    

Aig AigManager::aig_false()
{
    return aig_f_;
}


Aig AigManager::aig_var(int v)
{
    assert(v > 0);
    Aig_ av(v);
    return mk(insert(&av), false);
}


Aig AigManager::aig_nosimpl_and(Aig a1, Aig a2)
{
    if (aig_id(a1) > aig_id(a2)) {
        std::swap(a1, a2);
    }
    Aig_ a(a1, a2);
    return mk(insert(&a), false);
}


Aig AigManager::aig_not(Aig a)
{
    Aig_ *aa = aig_get(a);
    bool neg = aig_negated(a);
    return mk(aa, !neg);
}


AigManager::Aig_ *AigManager::insert(Aig_ *a)
{
    std::pair<AigStore::iterator, bool> p = store_.insert(*a);
    Aig_ *ret = const_cast<Aig_ *>(&(*(p.first)));
    if (p.second) {
        ret->id_ = next_id_++;
    }
    return ret;
}


void AigManager::aig_get_components(
    Aig a, bool *out_neg, Aig_ **out_aa,
    Aig *out_l, Aig_ **out_ll, bool *out_neg_l,
    Aig *out_r, Aig_ **out_rr, bool *out_neg_r)
{
    *out_neg = aig_negated(a);
    *out_l = aig_null();
    *out_r = aig_null();
    *out_ll = NULL;
    *out_rr = NULL;
    *out_neg_l = false;
    *out_neg_r = false;

    *out_aa = aig_get(a);
    if ((*out_aa)->right()) {
        *out_l = (*out_aa)->left();
        *out_ll = aig_get(*out_l);
        *out_neg_l = aig_negated(*out_l);

        *out_r = (*out_aa)->right();
        *out_rr = aig_get(*out_r);
        *out_neg_r = aig_negated(*out_r);
    } else {
        *out_l = a;
        *out_ll = *out_aa;
        *out_neg_l = *out_neg;
    }
}


Aig AigManager::simplify_1(Aig a, Aig b)
{
    if (a == b) {
        return a;
    } else if (a == aig_false() || b == aig_false()) {
        return aig_false();
    } else if (a == aig_true()) {
        return b;
    } else if (b == aig_true()) {
        return a;
    }
    return aig_null();
}


Aig AigManager::simplify_2(Aig a, Aig b)
{
    bool sa, sb;
    Aig_ *aa, *bb;
    Aig al, ar, bl, br;
    Aig_ *aal, *aar, *bbl, *bbr;
    bool sal, sar, sbl, sbr;

    aig_get_components(a, &sa, &aa, &al, &aal, &sal, &ar, &aar, &sar);
    aig_get_components(b, &sb, &bb, &bl, &bbl, &sbl, &br, &bbr, &sbr);

    if ((!sa &&
         ((aal == bb && sal != sb) ||
          (aar == bb && sar != sb))) ||
        (!sb &&
         ((bbl == aa && sbl != sa) ||
          (bbr == aa && sbr != sa)))) {
        // Contradiction A
        return aig_false();
    }

    if (!sa && !sb && aar && bbr &&
        ((aal == bbl && sal != sbl) ||
         (aal == bbr && sal != sbr) ||
         (aar == bbl && sar != sbl) ||
         (aar == bbr && sar != sbr))) {
        // Contradiction S
        return aig_false();
    }

    if (sa && !sb && !bbr &&
        ((aal == bb && sal != sb) ||
         (aar == bb && sar != sb))) {
        // Subsumption A
        return b;
    }
    
    if (sb && !sa && !aar &&
        ((bbl == aa && sbl != sa) ||
         (bbr == aa && sbr != sa))) {
        // Subsumption A
        return a;
    }

    if (sa && !sb && aar && bbr &&
        ((aal == bbl && sal != sbl) ||
         (aal == bbr && sal != sbr) ||
         (aar == bbl && sar != sbl) ||
         (aar == bbr && sar != sbr))) {
        // Subsumption S
        return b;
    }
    if (sb && !sa && aar && bbr &&
         ((bbl == aal && sbl != sal) ||
          (bbl == aar && sbl != sar) ||
          (bbr == aal && sbr != sal) ||
          (bbr == aar && sbr != sar))) {
        // Subsumption S
        return a;
    }

    if (!sa &&
         ((aal == bb && sal == sb) ||
          (aar == bb && sar == sb))) {
        // Idempotence
        return a;
    }
    if (!sb &&
         ((bbl == aa && sbl == sa) ||
          (bbr == aa && sbr == sa))) {
        // Idempotence
        return b;
    }

    if (sa && sb && aar && bbr) {
        // Resolution ?
        if (aal == bbl && sal == sbl && aar == bbr && sar != sbr) {
            return aig_not(al);
        } else if (aal == bbr && sal == sbr && aar == bbl && sar != sbl) {
            return aig_not(al);
        } else if (aar == bbl && sar == sbl && aal == bbr && sal != sbr) {
            return aig_not(ar);
        } else if (aar == bbr && sar == sbr && aal == bbl && sal != sbl) {
            return aig_not(ar);
        }
    }

    return aig_null();
}


bool AigManager::simplify_3(Aig &a, Aig &b)
{
    bool sa, sb;
    Aig_ *aa, *bb;
    Aig al, ar, bl, br;
    Aig_ *aal, *aar, *bbl, *bbr;
    bool sal, sar, sbl, sbr;

    aig_get_components(a, &sa, &aa, &al, &aal, &sal, &ar, &aar, &sar);
    aig_get_components(b, &sb, &bb, &bl, &bbl, &sbl, &br, &bbr, &sbr);

    if (sa && aar) {
        // Substitution A
        if (aar == bb && sar == sb) {
            a = aig_not(al);
            return true;
        } else if (aal == bb && sal == sb) {
            a = aig_not(ar);
            return true;
        }
    }
    
    if (sb && bbr) {
        // Substitution A
        if (bbr == aa && sbr == sa) {
            b = aig_not(bl);
            return true;
        } else if (bbl == aa && sbl == sa) {
            b = aig_not(br);
            return true;
        }
    }

    if (sa && !sb && aar && bbr) {
        // Substitution S
        if ((aar == bbl && sar == sbl) || (aar == bbr && sar == sbr)) {
            a = aig_not(al);
            return true;
        } else if ((aal == bbl && sal == sbl) || (aal == bbr && sal == sbr)) {
            a = aig_not(ar);
            return true;
        }
    }

    if (sb && !sa && aar && bbr) {
        // Substitution S
        if ((bbr == aal && sbr == sal) || (bbr == aar && sbr == sar)) {
            b = aig_not(bl);
            return true;
        } else if ((bbl == aal && sbl == sal) || (bbl == aar && sbl == sar)) {
            b = aig_not(br);
            return true;
        }
    }
    
    return false;
}


Aig AigManager::aig_and(Aig a, Aig b)
{
    Aig res = aig_null();

    while (true) {
        res = simplify_1(a, b);
        if (res != aig_null()) {
            break;
        }
        res = simplify_2(a, b);
        if (res != aig_null()) {
            break;
        }
        bool cont = simplify_3(a, b);
        if (!cont) {
            break;
        }
    }
    if (res == aig_null()) {
        res = aig_nosimpl_and(a, b);
    }

    return res;
}



std::string AigManager::aig2str(Aig a)
{
    bool neg = aig_negated(a);
    Aig_ *aa = aig_get(a);
    std::string ret = neg ? "(not " : "";
    if (aig_is_and(a)) {
        std::string s1 = aig2str(aa->left());
        std::string s2 = aig2str(aa->right());
        ret += "(and " + s1 + " " + s2 + ")";
    } else {
        std::ostringstream tmp;
        tmp << "v" << aa->var();
        ret += tmp.str();
    }
    if (neg) {
        ret += ")";
    }
    return ret;
}


void AigManager::find_parents(std::vector<Aig> &to_process, Aig a,
                              ParentsMap &pmap)
{
    HashSet<Aig> seen;
    
    to_process.push_back(a);
    pmap[a] = 0;
    while (!to_process.empty()) {
        Aig cur = to_process.back();
        bool neg = aig_negated(cur);
        Aig_ *aa = aig_get(cur);
        Aig curn = mk(aa, false);
        to_process.pop_back();

        if (!seen.insert(curn).second) {
            continue;
        }

        if (aa->right() != aig_null()) {
            Aig l = aa->left();
            Aig r = aa->right();
            to_process.push_back(l);
            to_process.push_back(r);

            pmap[mk(aig_get(l), false)] += 1;
            pmap[mk(aig_get(r), false)] += 1;
        }
    }
}


void AigManager::aig2aiger(Aig a, std::ostream &out)
{
    std::vector<Aig_ *> to_process;
    HashMap<Aig_ *, int> aig2idx;
    int curindex = 1;

    std::vector<Aig_ *> inputs;
    std::vector<Aig_ *> gates;

    // count the number of inputs and of gates
    to_process.push_back(aig_get(a));
    while (!to_process.empty()) {
        Aig_ *aa = to_process.back();
        if (aig2idx.find(aa) != aig2idx.end()) {
            to_process.pop_back();
            continue;
        }

        Aig l = aa->left();
        Aig r = aa->right();

        if (r == aig_null()) {
            inputs.push_back(aa);
            to_process.pop_back();
            aig2idx[aa] = curindex++;
        } else {
            bool children_done = true;
            Aig_ *ll = aig_get(l);
            Aig_ *rr = aig_get(r);
            if (aig2idx.find(ll) == aig2idx.end()) {
                to_process.push_back(ll);
                children_done = false;
            }
            if (aig2idx.find(rr) == aig2idx.end()) {
                to_process.push_back(rr);
                children_done = false;
            }
            if (children_done) {
                to_process.pop_back();
                gates.push_back(aa);
                aig2idx[aa] = curindex++;
            }
        }
    }

    // output the header
    out << "aag " << (curindex-1)
        << ' ' << inputs.size() << " 0 1 " << gates.size() << '\n';

    // write the inputs
    for (size_t i = 0; i < inputs.size(); ++i) {
        Aig_ *aa = inputs[i];
        int idx = aig2idx[aa];
        int v = idx * 2;
        out << v << '\n';
    }
    // write the output
    int idx = aig2idx[aig_get(a)];
    int v = idx * 2;
    if (aig_negated(a)) {
        ++v;
    }
    out << v << '\n';
    // now write the gates
    for (size_t i = 0; i < gates.size(); ++i) {
        Aig_ *aa = gates[i];
        idx = aig2idx[aa];
        v = idx * 2;
        out << v << ' ';
        Aig l = aa->left();
        Aig r = aa->right();
        v = aig2idx[aig_get(l)] * 2;
        if (aig_negated(l)) {
            ++v;
        }
        out << v << ' ';
        v = aig2idx[aig_get(r)] * 2;
        if (aig_negated(r)) {
            ++v;
        }
        out << v << '\n';
    }
}


//-----------------------------------------------------------------------------
// CNF conversion with logic synthesis (a-la ABC)
//-----------------------------------------------------------------------------
// code derived from ABC by Alan Mishchenko. Copyright of original code follows
//
// ABC: System for Sequential Synthesis and Verification
// 
// http://www.eecs.berkeley.edu/~alanmi/abc/
// 
// Copyright (c) The Regents of the University of California. All rights
// reserved.
// 
// Permission is hereby granted, without written agreement and without license
// or royalty fees, to use, copy, modify, and distribute this software and its
// documentation for any purpose, provided that the above copyright notice and
// the following two paragraphs appear in all copies of this software.
// 
// IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
// DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
// OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY
// OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
// AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON
// AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
// PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

bool AigManager::filter_cut(const Cut &c, std::vector<Cut> &cuts)
{
    for (int i = 0; i < int(cuts.size()); ++i) {
        Cut &cc = cuts[i];
        if (cc.num_leaves > c.num_leaves) {
            // skip the non-contained cuts
            if ((cc.abstraction & c.abstraction) != c.abstraction) {
                continue;
            }
            // check containment precisely
            if (cut_check_dominance(c, cc)) {
                // remove cc from the cuts
                std::swap(cc, cuts.back());
                cuts.pop_back();
                --i;
            }
        } else {
            // skip the non-contained cuts
            if ((c.abstraction & cc.abstraction) != cc.abstraction) {
                continue;
            }
            // check containment precisely
            if (cut_check_dominance(cc, c)) {
                // c is redundant
                return true;
            }            
        }
    }

    return false;
}


inline bool AigManager::cut_check_dominance(const Cut &c1, const Cut &c2)
{
    for (int i = 0; i < c1.num_leaves; ++i) {
        int j = 0;
        for ( ; j < c2.num_leaves; ++j) {
            if (c1.leaves[i] == c2.leaves[j]) {
                break;
            }
        }
        if (j == c2.num_leaves) {
            return false;
        }
    }
    return true;
}


bool AigManager::cut_minimize_support(Cut &c, bool allow_const_nodes)
{
    TruthTable masks[4][2] = {
        { 0x5555, 0xAAAA },
        { 0x3333, 0xCCCC },
        { 0x0F0F, 0xF0F0 },
        { 0x00FF, 0xFF00 }
    };
    uint16_t phase = 0;
    int nl = c.num_leaves;
    // compute the support of the cut's function
    for (int i = 0; i < c.num_leaves; ++i) {
        if ((c.truth & masks[i][0]) == ((c.truth & masks[i][1]) >> (1 << i))) {
            --nl;
        } else {
            phase |= (1 << i);
        }
    }
    if (nl == c.num_leaves) {
        return false;
    }

    if (nl == 0 && !allow_const_nodes) {
        return false;
    }
    
    // shrink the truth table
    c.truth = truth_shrink(c.truth, c.num_leaves, phase);
    // update leaves and abstraction
    c.abstraction = 0;
    int k = 0;
    for (int i = 0; i < c.num_leaves; ++i) {
        if (!(phase & (1 << i))) {
            continue;
        }
        c.leaves[k++] = c.leaves[i];
        c.abstraction |= cut_calc_abstraction(c.leaves[i]);
    }
    assert(k == nl);
    c.num_leaves = nl;
    return true;
}


bool AigManager::merge_cuts(const Cut &c1, const Cut &c2, Cut &out)
{
    const Cut *a = &c1, *b = &c2;
    if (a->num_leaves < b->num_leaves) {
        std::swap(a, b);
    }

    if (a->num_leaves == 4 && b->num_leaves == 4) {
        if (a->abstraction != b->abstraction) {
            return false;
        }
        for (int i = 0; i < 4; ++i) {
            if (a->leaves[i] != b->leaves[i]) {
                return false;
            } else {
                out.leaves[i] = a->leaves[i];
            }
        }
        out.num_leaves = 4;
    } else if (a->num_leaves == 4) {
        if ((a->abstraction & b->abstraction) != b->abstraction) {
            return false;
        }
        for (int i = 0; i < b->num_leaves; ++i) {
            int k = 3;
            for (; k >= 0; --k) {
                if (a->leaves[k] == b->leaves[i]) {
                    break;
                }
            }
            if (k == -1) { // did not find
                return false;
            }
        }
        for (int i = 0; i < 4; ++i) {
            out.leaves[i] = a->leaves[i];
        }
        out.num_leaves = 4;
    } else {
        // general case
        int i = 0, k = 0;
        int c = 0;
        for ( ; c < 4; ++c) {
            if (k == b->num_leaves) {
                if (i == a->num_leaves) {
                    out.num_leaves = c;
                    k = b->num_leaves;
                    break;
                }
                out.leaves[c] = a->leaves[i++];
                continue;
            }
            if (i == a->num_leaves) {
                if (k == b->num_leaves) {
                    out.num_leaves = c;
                    i = a->num_leaves;
                    break;
                }
                out.leaves[c] = b->leaves[k++];
                continue;
            }
            
            assert(a->leaves[i]);
            assert(b->leaves[k]);
            
            if (aig_id(a->leaves[i]) > aig_id(b->leaves[k])) {
                out.leaves[c] = a->leaves[i++];
                continue;
            }
            if (aig_id(a->leaves[i]) < aig_id(b->leaves[k])) {
                out.leaves[c] = b->leaves[k++];
                continue;
            }
            out.leaves[c] = a->leaves[i++]; 
            k++;
        }
        if (i < a->num_leaves || k < b->num_leaves) {
            return false;
        }
        out.num_leaves = c;
    }

    out.abstraction = a->abstraction | b->abstraction;    
    
    return true;
}


AigManager::TruthTable AigManager::cut_compute_truth_table(
    const Cut &p, const Cut &c0, bool neg0, const Cut &c1, bool neg1)
{
    TruthTable t0 = neg0 ? ~(c0.truth) : c0.truth;
    TruthTable t1 = neg1 ? ~(c1.truth) : c1.truth;
    uint16_t p0 = truth_compute_phase(p, c0);
    uint16_t p1 = truth_compute_phase(p, c1);
    t0 = truth_stretch(t0, c0.num_leaves, p0);
    t1 = truth_stretch(t1, c1.num_leaves, p1);
    return t0 & t1;
}


inline AigManager::TruthTable AigManager::truth_stretch(
    TruthTable truth, int num_vars, uint16_t phase)
{
    int var = num_vars - 1;
    for (int i = 3; i >= 0; --i) {
        if (phase & (1 << i)) {
            for (int k = var; k < i; ++k) {
                truth = truth_swap_adjacent_vars(truth, k);
            }
            var--;
        }
    }
    assert(var == -1);
    return truth;
}


inline AigManager::TruthTable AigManager::truth_shrink(
    TruthTable truth, int num_vars, uint16_t phase)
{
    int var = 0;
    for (int i = 0; i < 4; ++i) {
        if (phase & (1 << i)) {
            for (int k = i-1; k >= var; --k) {
                truth = truth_swap_adjacent_vars(truth, k);
            }
            var++;
        }
    }
    return truth;
}


inline AigManager::TruthTable AigManager::truth_swap_adjacent_vars(
    TruthTable truth, int var)
{
    assert(var >= 0 && var <= 2);

    switch (var) {
    case 0:
        return ((truth & 0x99999999) | ((truth & 0x22222222) << 1) |
                ((truth & 0x44444444) >> 1));
    case 1:
        return ((truth & 0xC3C3C3C3) | ((truth & 0x0C0C0C0C) << 2) |
                ((truth & 0x30303030) >> 2));
    case 2:
        return ((truth & 0xF00FF00F) | ((truth & 0x00F000F0) << 4) |
                ((truth & 0x0F000F00) >> 4));
    default:
        assert(false);
    }
    return 0;
}


inline uint16_t AigManager::truth_compute_phase(const Cut &p, const Cut &c)
{
    uint16_t phase = 0;
    int i, k;
    for (i = k = 0; i < p.num_leaves; ++i) {
        if (k == c.num_leaves) {
            break;
        }
        if (aig_id(p.leaves[i]) > aig_id(c.leaves[k])) {
            continue;
        }
        assert(p.leaves[i] == c.leaves[k]);
        phase |= (1 << i);
        k++;
    }
    return phase;    
}


inline uint32_t AigManager::get_area_flow_cost(
    const Cut &c, const ParentsMap &pmap, const AreaFlowCostMap &afmap,
    const SopData &sopdata, uint32_t &out_val)
{
    uint32_t cost = 10 * get_num_clauses_cut(c, sopdata);
    out_val = 0;
    for (int i = 0; i < c.num_leaves; ++i) {
        ParentsMap::const_iterator it = pmap.find(c.leaves[i]);
        assert(it != pmap.end());
        int nrefs = it->second;
        out_val += nrefs;
        AreaFlowCostMap::const_iterator it2 = afmap.find(c.leaves[i]);
        assert(it2 != afmap.end());
        cost += it2->second / (nrefs ? nrefs : 1);
    }
    return cost;
}


inline uint32_t AigManager::get_num_clauses_cut(const Cut &c,
                                                const SopData &sopdata)
{
    assert(size_t(c.truth) < sopdata.size());
    assert(size_t(0xFFFF & ~c.truth) < sopdata.size());

    return sopdata[c.truth].size() + sopdata[0xFFFF & ~c.truth].size();
}


void AigManager::cnf_init_sop_data(SopData &out)
{
    TruthTable masks[4][2] = {
        { 0x5555 /* 0101010101010101 */, 0xAAAA /* 1010101010101010 */ },
        { 0x3333 /* 0011001100110011 */, 0xCCCC /* 1100110011001100 */ },
        { 0x0F0F /* 0000111100001111 */, 0xF0F0 /* 1111000011110000 */ },
        { 0x00FF /* 0000000011111111 */, 0xFF00 /* 1111111100000000 */ }
    };
    char transtbl[256];

    // fill the translation table
    for (int i = 0; i < 256; ++i) {
        transtbl[i] = char(-1);
    }
    for (int i = 0; i < 81; ++i) {
        transtbl[int(aigsopdata::translation_table[i])] = char(i);
    }

    // load the data from the table
    std::vector<char> buf;
    out.reserve(65600); // leave room for the final padding in aigsopdata::data
    out.push_back(buf);
    for (size_t j = 0; aigsopdata::data[j]; ++j) {
        for (const char *c = aigsopdata::data[j]; *c; ++c) {
            if (*c == ' ') {
                out.push_back(buf);
                buf.clear();
            } else {
                buf.push_back(transtbl[int(*c)]);
            }
        }
    }

#ifndef NDEBUG
    // verify the results - derive truth table from SOP
    for (int i = 1; i < 65536; ++i) {
        TruthTable truth = 0;
        TruthTable cube;
        int lit;
        for (size_t k = 0; k < out[i].size(); ++k) {
            cube = 0xFFFF;
            lit = out[i][k];
            for (int b = 3; b >= 0; --b) {
                if (lit % 3 == 0) {
                    cube &= masks[b][0];
                } else if (lit % 3 == 1) {
                    cube &= masks[b][1];
                }
                lit = lit / 3;
            }
            truth |= cube;
        }
        assert(truth == i);
    }
#endif
}


void AigManager::compute_best_cuts(std::vector<Aig> &to_process, Aig a,
                                   CutsMap &cuts, const ParentsMap &pmap,
                                   const SopData &sopdata)
{
    AreaFlowCostMap afmap;

    Aig aa = mk(aig_get(a), false);
    to_process.push_back(aa);
    
    while (!to_process.empty()) {
        Aig cur = to_process.back();

        if (afmap.find(cur) != afmap.end()) {
            to_process.pop_back();
            continue;
        }

        Aig_ *c = aig_get(cur);
        Aig l = c->left();
        Aig r = c->right();

        if (r == aig_null()) {
            afmap[cur] = 0;
            to_process.pop_back();
            continue;
        }
        assert(cuts.find(cur) != cuts.end());
        
        if (cuts[cur].size() == 1) {
            afmap[cur] = 0;
            to_process.pop_back();
            continue;
        }
       
        Aig ll = mk(aig_get(l), false);
        Aig rr = mk(aig_get(r), false);

        bool children_done = true;
        if (afmap.find(ll) == afmap.end()) {
            children_done = false;
            to_process.push_back(ll);
        }
        if (afmap.find(rr) == afmap.end()) {
            children_done = false;
            to_process.push_back(rr);
        }

        if (children_done) {
            to_process.pop_back();

            std::vector<Cut> &vc = cuts[cur];
            Cut *best = NULL;
            uint32_t bestcost = 0;
            uint32_t bestval = 0;
            for (size_t i = 1 /* skip the trivial cut */; i < vc.size(); ++i) {
                uint32_t val = 0;
                uint32_t cost =
                    get_area_flow_cost(vc[i], pmap, afmap, sopdata, val);
                if (!best || bestcost > cost ||
                    (bestcost == cost && bestval < val)) {
                    best = &vc[i];
                    bestcost = cost;
                    bestval = val;
                }
            }
            if (!best) {
                // check the first cut, it must be either true or false
                assert(vc[0].num_leaves == 0);
                assert(vc[0].truth == 0 || vc[0].truth == 0xFFFF);
                best = &(vc[0]);
                bestcost = 0;
            }
            assert(best);
            std::swap(*best, vc.front());
            afmap[cur] = bestcost;
        }
    }
}


namespace {

struct RevAigId_lt {
    RevAigId_lt(AigManager &mgr): mgr_(mgr) {}
    
    bool operator()(Aig a, Aig b) const
    {
        return mgr_.aig_id(a) > mgr_.aig_id(b);
    }
    
    AigManager &mgr_;
};

} // namespace


void AigManager::get_nodes_in_mapping(
    std::vector<Aig> &to_process, Aig a,
    const CutsMap &cuts, const ParentsMap &pmap, std::vector<Aig> &out)
{
    HashSet<Aig> seen;
    Aig aa = mk(aig_get(a), false);
    to_process.push_back(aa);

    while (!to_process.empty()) {
        Aig cur = to_process.back();
        to_process.pop_back();
        
        if (aig_is_and(cur) && seen.insert(cur).second) {
            out.push_back(cur);

            CutsMap::const_iterator it = cuts.find(cur);
            assert(it != cuts.end());
            assert(!it->second.empty());

            const Cut &c = it->second.front();
            for (int i = 0; i < c.num_leaves; ++i) {
                Aig l = c.leaves[i];
                Aig ll = mk(aig_get(l), false);
                to_process.push_back(ll);
            }
        }
    }

    std::sort(out.begin(), out.end(), RevAigId_lt(*this));
}


} // namespace simplic3
