// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// And-inverter graphs with local 2-level rewriting
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_AIG_H
#define SIMPLIC3_AIG_H

#include "simplic3/hashmap.h"
#include <vector>
#include <stdint.h>
#include <string>
#include <ostream>

namespace simplic3 {

typedef uintptr_t Aig;

class AigManager {
public:
    AigManager();
    ~AigManager();

    Aig aig_true();
    Aig aig_false();
    Aig aig_var(int v);
    Aig aig_and(Aig a1, Aig a2);
    static Aig aig_not(Aig a);
    Aig aig_or(Aig a, Aig b);
    Aig aig_ite(Aig c, Aig t, Aig e);
    Aig aig_iff(Aig a, Aig b);
    Aig aig_xor(Aig a, Aig b);
    static Aig aig_null() { return 0; }
    static size_t aig_id(Aig a) { return aig_get(a)->id_; }
    static bool aig_is_negated(Aig a) { return aig_negated(a); }
    static bool aig_is_and(Aig a) { return aig_get(a)->right() != aig_null(); }
    static int aig_get_var(Aig a) { return aig_get(a)->var(); }
    static Aig aig_get_left(Aig a) { return aig_get(a)->left(); }
    static Aig aig_get_right(Aig a) { return aig_get(a)->right(); }
    static Aig aig_strip(Aig a) { return reinterpret_cast<Aig>(aig_get(a)); }

    template <class Sink>
    void aig2cnf(Aig a, Sink &sink);

    template <class Sink>
    void aig2cnf_ls(Aig a, Sink &sink,
                    size_t max_cuts=0, bool use_cut_simplification=true);
    ///< CNF conversion via logic synthesis
    
    std::string aig2str(Aig a);
    void aig2aiger(Aig a, std::ostream &out);

private:
    AigManager(const AigManager &);
    AigManager &operator=(const AigManager &);
    
    struct Aig_ {
    public:
        Aig_(Aig lhs=0, Aig rhs=0):
            id_(0), lhs_(lhs), rhs_(rhs) {}
        Aig_(int v): id_(0), v_(v), rhs_(0) {}

        int var() const { return v_; }
        Aig left() const { return lhs_; }
        Aig right() const { return rhs_; }
    
    public:
        size_t id_;
        union {
            intptr_t v_;
            Aig lhs_;
        };
        Aig rhs_;    
    };

    static bool aig_negated(Aig a) { return bool(a & Aig(1)); }
    static Aig_ *aig_get(Aig a)
    { return reinterpret_cast<Aig_ *>(a & ~Aig(1)); }
    
    static size_t aig_hash(const Aig_ &a)
    {
        static const size_t halfsz = sizeof(size_t) * 4;
        return (a.right() << halfsz) + a.left();
    }

    static bool aig_eq(const Aig_ &a1, const Aig_ &a2)
    {
        return (a1.left() == a2.left()) && (a1.right() == a2.right());
    }    

    static Aig mk(Aig_ *a, bool neg)
    { return reinterpret_cast<Aig>(a) | Aig(neg); }

    Aig_ *insert(Aig_ *a);

    void aig_get_components(Aig a, bool *out_neg, Aig_ **out_aa,
                            Aig *out_l, Aig_ **out_ll, bool *out_neg_l,
                            Aig *out_r, Aig_ **out_rr, bool *out_neg_r);
    Aig simplify_1(Aig a, Aig b);
    Aig simplify_2(Aig a, Aig b);
    bool simplify_3(Aig &a, Aig &b);

    Aig aig_nosimpl_and(Aig a, Aig b);

    // auxiliary methods for aig2cnf
    typedef HashMap<Aig, int> ParentsMap;    
    void find_parents(std::vector<Aig> &to_process, Aig a, ParentsMap &pmap);
    template <class Sink>
    bool do_collapse(Aig cur, Sink &sink, std::vector<Aig> &to_process,
                     std::vector<Aig> &tmp, ParentsMap &pmap);
    bool is_shared(Aig a, ParentsMap &pmap);

    //-------------------------------------------------------------------------
    // cnf conversion with logic synthesis (a-la ABC)
    //-------------------------------------------------------------------------
    typedef uint16_t TruthTable;
    ///< Truth tables are represented, as in ABC, as a bit field, where the
    ///< bit in position i represents the output for the inputs represented by
    ///< the integer i, which is the value of the bit-field of the inputs. For
    ///< example, the input 0010 corresponds to index 2, 0011 to 3, and 0101 to
    ///< 5. For functions with k inputs, the table is stored in the first
    ///< 2^k bits. However, the bit-pattern is replicated to fill the whole 16
    ///< bits, in order to make the manipulations of truth tables (such as
    ///< the computation of the table for an Aig node given the tables of the
    ///< children) easier. For example, the truth table for a single variable
    ///< consists of 8 copies of the bit pattern 10, i.e. 1010101010101010
    
    struct Cut {
        TruthTable truth;
        int16_t num_leaves : 16;
        uint32_t abstraction; // an abstraction of this cut, for speeding up
                              // some computations
        Aig leaves[4];
    };
    
    typedef HashMap<Aig, std::vector<Cut> > CutsMap;
    typedef std::vector<int> Cover;
    typedef HashMap<Aig, uint32_t> AreaFlowCostMap;
    typedef std::vector< std::vector<char> > SopData;

    // cut manipulation
    template <class Sink>
    void find_cuts(std::vector<Aig> &to_process, Aig a, CutsMap &cuts,
                   const ParentsMap &pmap, size_t max_cuts, bool simpl,
                   Sink &sink);
    void init_cut(Aig a, Cut &c);
    static bool merge_cuts(const Cut &c1, const Cut &c2, Cut &out);
    static uint32_t cut_calc_abstraction(Aig a)
    { return uint32_t(1 << (a & 31)); }
    static bool filter_cut(const Cut &c, std::vector<Cut> &cuts);
    static bool cut_minimize_support(Cut &c, bool allow_const_nodes);
    static bool cut_check_dominance(const Cut &c1, const Cut &c2);

    // truth-table manipulation
    static TruthTable cut_compute_truth_table(const Cut &p,
                                              const Cut &c0, bool neg0,
                                              const Cut &c1, bool neg1);
    static TruthTable truth_swap_adjacent_vars(TruthTable truth, int var);
    static TruthTable truth_stretch(TruthTable truth, int num_vars,
                                    uint16_t phase);
    static TruthTable truth_shrink(TruthTable truth, int num_vars,
                                   uint16_t phase);
    static uint16_t truth_compute_phase(const Cut &p, const Cut &c);

    // cost for cuts
    static uint32_t get_area_flow_cost(const Cut &c, const ParentsMap &pmap,
                                       const AreaFlowCostMap &afmap,
                                       const SopData &sopdata,
                                       uint32_t &out_val);
    static uint32_t get_num_clauses_cut(const Cut &c, const SopData &sopdata);
    void compute_best_cuts(std::vector<Aig> &to_process, Aig a,
                           CutsMap &cuts, const ParentsMap &pmap,
                           const SopData &sopdata);
    void get_nodes_in_mapping(std::vector<Aig> &to_process, Aig a,
                              const CutsMap &cuts, const ParentsMap &pmap,
                              std::vector<Aig> &out);

    // loading the pre-computed SOPs
    static void cnf_init_sop_data(SopData &out);
    //-------------------------------------------------------------------------
    

    friend struct Aig_HashEq;
    struct Aig_HashEq {
        size_t operator()(Aig a) const
        {
            Aig_ *aa = AigManager::aig_get(a);
            size_t h = AigManager::aig_hash(*aa);
            h <<= size_t(1);
            size_t n = size_t(AigManager::aig_negated(a));
            return (h | n);
        }

        bool operator()(Aig a1, Aig a2) const
        {
            Aig_ *aa1 = AigManager::aig_get(a1);
            Aig_ *aa2 = AigManager::aig_get(a2);
            bool s1 = AigManager::aig_negated(a1);
            bool s2 = AigManager::aig_negated(a2);

            if (s1 == s2) {
                return (aa1 == aa2) || (AigManager::aig_eq(*aa1, *aa2));
            } else {
                return false;
            }
        }
    };

    friend struct AigStore_HashEq;
    struct AigStore_HashEq {
        size_t operator()(const Aig_ &a) const
        {
            return AigManager::aig_hash(a);
        }

        bool operator()(const Aig_ &a1, const Aig_ &a2) const
        {
            return AigManager::aig_eq(a1, a2);
        }
    };
    
    typedef HashSet<Aig_, AigStore_HashEq, AigStore_HashEq> AigStore;
    AigStore store_;
    size_t next_id_;
    Aig aig_t_;
    Aig aig_f_;

    SopData *sopdata_; ///< data for CNF conversion via logic synthesis    
};


inline Aig AigManager::aig_or(Aig a, Aig b)
{ return aig_not(aig_and(aig_not(a), aig_not(b))); }

inline Aig AigManager::aig_ite(Aig c, Aig t, Aig e)
{ return aig_or(aig_and(c, t), aig_and(aig_not(c), e)); }

inline Aig AigManager::aig_iff(Aig a, Aig b)
{ return aig_or(aig_and(a, b), aig_and(aig_not(a), aig_not(b))); }

inline Aig AigManager::aig_xor(Aig a, Aig b)
{ return aig_not(aig_iff(a, b)); }



/**
 * Interface that Sinks for converting AIGs to CNF must implement. This class
 * is for documentation purposes only
 */
class AigSinkInterface {
public:
    virtual ~AigSinkInterface() {}
    virtual bool has_label(Aig a) = 0;
    virtual void set_label(Aig a) = 0;
    virtual void set_label(Aig a, int var) = 0;
    virtual void begin_clause() = 0;
    virtual void add_literal(Aig a, bool negated) = 0;
    virtual void end_clause() = 0;
    virtual void toplevel_literal(Aig a, bool negated) = 0;
};


} // namespace simplic3

#endif // SIMPLIC3_AIG_H
