// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// And-inverter graphs with local 2-level rewriting
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "simplic3/aig.h"
#include <sstream>
#include <assert.h>

namespace simplic3 {

inline bool AigManager::is_shared(Aig a, ParentsMap &pmap)
{
    ParentsMap::iterator it = pmap.find(a);
    if (it == pmap.end() || it->second <= 1) {
        return false;
    }
    return true;
}


template <class Sink>
bool AigManager::do_collapse(Aig cur, Sink &sink, std::vector<Aig> &to_process,
                             std::vector<Aig> &tmp, ParentsMap &pmap)
{
    tmp.clear();
    Aig_ *cura = aig_get(cur);
    tmp.push_back(cura->left());
    tmp.push_back(cura->right());
    bool children_done = true;
    while (!tmp.empty()) {
        Aig a = tmp.back();
        tmp.pop_back();
        Aig_ *aa = aig_get(a);
        Aig an = mk(aa, false);
        if (!sink.has_label(an)) {
            if (aa->right() == aig_null() || (a != an) || is_shared(an, pmap)) {
                to_process.push_back(a);
                children_done = false;
            } else {
                assert(aa->left() != aig_null());
                assert(aa->right() != aig_null());

                tmp.push_back(aa->left());
                tmp.push_back(aa->right());
            }
        }
    }
    return children_done;
}


template <class Sink>
void AigManager::aig2cnf(Aig a, Sink &sink)
{
    std::vector<Aig> to_process, to_process_2, args;
    ParentsMap pmap;

    find_parents(to_process, a, pmap);
    to_process.push_back(a);

    while (!to_process.empty()) {
        Aig cur = to_process.back();
        bool neg = aig_negated(cur);
        Aig_ *aa = aig_get(cur);
        Aig curn = mk(aa, false);

        assert(cur != aig_true());
        assert(cur != aig_false());

        if (sink.has_label(curn)) {
            to_process.pop_back();
            continue;
        }

        Aig l = aa->left();
        Aig r = aa->right();

        if (r != aig_null()) {
            bool children_done = true;
            children_done =
                do_collapse(curn, sink, to_process, to_process_2, pmap);
            
            if (children_done) {
                // try to collapse multiple ANDs together,
                // to create less variables...
                to_process_2.clear();
                args.clear();
                to_process_2.push_back(curn);
                while (!to_process_2.empty()) {
                    Aig a = to_process_2.back();
                    Aig_ *pa = aig_get(a);
                    Aig an = mk(pa, false);
                    to_process_2.pop_back();
                    if (!sink.has_label(an) && !aig_negated(a) &&
                        pa->right() != aig_null() &&
                        (a == curn || !is_shared(a, pmap))) {
                        to_process_2.push_back(pa->left());
                        to_process_2.push_back(pa->right());
                    } else {
                        assert(sink.has_label(an));
                        args.push_back(a);
                    }
                }

                assert(args.size() >= 2);

                sink.set_label(curn);
                
                for (size_t i = 0; i < args.size(); ++i) {
                    sink.begin_clause();
                    sink.add_literal(curn, true);
                    Aig c = args[i];
                    sink.add_literal(mk(aig_get(c), false), aig_negated(c));
                    sink.end_clause();
                }

                sink.begin_clause();
                sink.add_literal(curn, false);
                for (size_t i = 0; i < args.size(); ++i) {
                    Aig c = args[i];
                    sink.add_literal(mk(aig_get(c), false), !aig_negated(c));
                }
                sink.end_clause();
            }
        } else {
            sink.set_label(curn, aa->var());
        }
    }

    Aig an = mk(aig_get(a), false);

    assert(sink.has_label(an));

    sink.toplevel_literal(an, aig_negated(a));
}


//-----------------------------------------------------------------------------
// CNF conversion with logic synthesis (a-la ABC)
//-----------------------------------------------------------------------------
// code derived from ABC by Alan Mishchenko. Copyright of original code follows
//
// ABC: System for Sequential Synthesis and Verification
// 
// http://www.eecs.berkeley.edu/~alanmi/abc/
// 
// Copyright (c) The Regents of the University of California. All rights
// reserved.
// 
// Permission is hereby granted, without written agreement and without license
// or royalty fees, to use, copy, modify, and distribute this software and its
// documentation for any purpose, provided that the above copyright notice and
// the following two paragraphs appear in all copies of this software.
// 
// IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
// DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
// OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY
// OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
// AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON
// AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
// PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

template <class Sink>
void AigManager::find_cuts(std::vector<Aig> &to_process, Aig a, CutsMap &cuts,
                           const ParentsMap &pmap, size_t max_cuts, bool simpl,
                           Sink &sink)
{
    if (max_cuts == 0) {
        max_cuts = 10;
        // size_t sz = pmap.size();
        // if (sz > 1000000) {
        //     max_cuts = 4;
        // } else if (sz > 500000) {
        //     max_cuts = 5;
        // } else if (sz > 100000) {
        //     max_cuts = 8;
        // }
    }
    
    // set the cut for the true node
    Cut c;
    init_cut(aig_true(), c);
    cuts[aig_true()].push_back(c);

    Aig an = mk(aig_get(a), false);
    to_process.push_back(an);
    
    while (!to_process.empty()) {
        Aig cur = to_process.back();
        assert(!aig_is_negated(cur));
        
        Aig_ *aa = aig_get(cur);

        CutsMap::iterator it = cuts.find(cur);
        if (it != cuts.end()) {
            to_process.pop_back();
            continue;
        }

        bool isleaf = (aa->right() == aig_null()) || sink.has_label(cur);
        if (isleaf) {
            init_cut(cur, c);
            cuts[cur].push_back(c);
            to_process.pop_back();
            continue;
        }

        bool children_done = true;
        Aig ll = aa->left();
        Aig rr = aa->right();
        Aig l = mk(aig_get(ll), false);
        Aig r = mk(aig_get(rr), false);

        if (cuts.find(l) == cuts.end()) {
            to_process.push_back(l);
            children_done = false;
        }
        if (cuts.find(r) == cuts.end()) {
            to_process.push_back(r);
            children_done = false;
        }

        if (children_done) {
            to_process.pop_back();

            // add the trivial cut first
            init_cut(cur, c);
            std::vector<Cut> &vc = cuts[cur];
            vc.reserve(max_cuts);
            vc.push_back(c);

            const std::vector<Cut> &vcl = cuts[l];
            const std::vector<Cut> &vcr = cuts[r];
            // then, create the other cuts by merging those of the children
            for (size_t i = 0; i < vcl.size(); ++i) {
                const Cut &cl = vcl[i];
                for (size_t j = 0; j < vcr.size(); ++j) {
                    const Cut &cr = vcr[j];

                    if (merge_cuts(cl, cr, c)) {
                        // check if the new cut is redundant, and remove cuts
                        // that are subsumed by this one
                        if (simpl && filter_cut(c, vc)) {
                            continue;
                        }
                        // compute the truth table
                        TruthTable tt =
                            cut_compute_truth_table(c, cl, aig_negated(ll),
                                                    cr, aig_negated(rr));
                        c.truth = tt;
                        // minimize the support of this cut
                        if (simpl && cut_minimize_support(c, true)) {
                            // if successful, check whether some of the
                            // previous cuts are now redundant
                            bool b = filter_cut(c, vc);
                            assert(!b);
                        }
                    } else {
                        continue;
                    }

                    vc.push_back(c);

                    if (vc.size() == max_cuts) {
                        goto nextcut;
                    }
                }
            }
          nextcut: ;
        }
    }
}


inline void AigManager::init_cut(Aig a, Cut &c)
{
    if (a == aig_true()) {
        c.num_leaves = 0;
        c.abstraction = 0;
        c.truth = 0xFFFF; // 0b1111111111111111
    } else {
        c.num_leaves = 1;
        c.leaves[0] = a;
        c.abstraction = cut_calc_abstraction(a);
        c.truth = 0xAAAA; // 0b1010101010101010
    }
}


template <class Sink>
void AigManager::aig2cnf_ls(Aig a, Sink &sink,
                            size_t max_cuts, bool use_cut_simplification)
{
    std::vector<Aig> to_process;
    std::vector<Aig> mapping;
    ParentsMap pmap;
    CutsMap cuts;

    if (!sopdata_) {
        sopdata_ = new SopData();
        cnf_init_sop_data(*sopdata_);
    }
    SopData &sopdata = *sopdata_;
    
    find_parents(to_process, a, pmap);
    find_cuts(to_process, a, cuts, pmap, max_cuts, use_cut_simplification,
              sink);
    compute_best_cuts(to_process, a, cuts, pmap, sopdata);
    get_nodes_in_mapping(to_process, a, cuts, pmap, mapping);

    for (size_t i = mapping.size(); i > 0; --i) {
        Aig cur = mapping[i-1];
        assert(aig_is_and(cur));
        assert(!aig_is_negated(cur));

        CutsMap::iterator it = cuts.find(cur);
        assert(it != cuts.end());
        assert(!it->second.empty());

        const Cut &c = it->second.front();

        if (sink.has_label(cur)) {
            continue;
        }

        sink.set_label(cur);
        
        if (c.num_leaves == 0) {
            // special case: cur is equivalent to false or true
            assert(c.truth == 0 || c.truth == 0xFFFF);
            sink.begin_clause();
            sink.add_literal(cur, c.truth == 0);
            sink.end_clause();
            continue;
        }

        for (int polarity = 0; polarity < 2; ++polarity) {
            TruthTable t = (polarity == 0 ? 0xFFFF & ~c.truth : c.truth);
            assert(t < sopdata.size());
            
            const std::vector<char> &sop = sopdata[t];
            for (size_t j = 0; j < sop.size(); ++j) {
                int lit = sop[j];
                sink.begin_clause();
                sink.add_literal(cur, polarity == 0);
                for (int b = 3; b >= 0; --b) {
                    int r = lit % 3;
                    lit /= 3;
                    if (r == 2) {
                        continue;
                    }
                    Aig l = c.leaves[b];
                    assert(!aig_is_negated(l));
                    
                    if (!sink.has_label(l)) {
                        assert(!aig_is_and(l));
                        sink.set_label(l, aig_get_var(l));
                    }
                    bool neg = (r == 1);
                    sink.add_literal(l, neg);
                }
                sink.end_clause();
            }
        }
    }

    Aig an = mk(aig_get(a), false);

    if (!sink.has_label(an)) {
        assert(!aig_is_and(an));
        sink.set_label(an, aig_get_var(an));
    }

    sink.toplevel_literal(an, aig_is_negated(a));
}

} // namespace simplic3
