// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// BMC engine
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "bmc.h"
#include <sstream>
#include <iomanip>

namespace simplic3 {

BMC::BMC(Options opts, Model *model, size_t prop_idx):
    opts_(opts),
    model_(model),
    prop_idx_(prop_idx),
    un_(model_),
    cb_(NULL)
{
    factory_ = NULL;
    verb_ = opts_.verbosity;
    min_k_ = opts_.bmc_mindepth;
    max_k_ = opts_.bmc_maxdepth;

    reached_depth_ = 0;
    total_time_ = 0.0;
    clausify_time_ = 0.0;
    preprocess_time_ = 0.0;
}


BMC::~BMC()
{
}


void BMC::set_safe_bound_callback(const SafeBoundCallback *cb)
{
    cb_ = cb;
}


Prover::Result BMC::prove()
{
    Aig bad = AigManager::aig_not(model_->properties()[prop_idx_]);
    return do_prove(bad);
}


Prover::Result BMC::do_prove(Aig target)
{
    TimeKeeper t(total_time_);

    factory_ = SatSolverFactory::get(opts_.solver);
    SatSolver *solver = factory_->new_satsolver();
    CnfConv cnf(model_, solver, opts_.cnf_simple, true);

    cex_trace_.clear();

    bool done = false;

    SatLitList cls;
    SatLitList assumptions;
    assumptions.push_back(satLit_Undef);
    
    // send initial states
    {
        TimeKeeper tk(clausify_time_);
        
        for (AigList::const_iterator it = model_->statevars().begin(),
                 end = model_->statevars().end(); it != end; ++it) {
            Aig v = *it;
            Aig a = un_.unroll(model_->init_formula(v), 0);
            SatLit l = cnf.clausify(a);

            cls.clear();
            cls.push_back(l);
            solver->add_clause(cls);
        }
    }

    reached_depth_ = 0;
    
    for (int k = min_k_; (max_k_ > 0 ? (k <= max_k_) : true) && !done; ++k) {
        if (verb_ > 0) {
            std::cout << "checking bound " << k << std::endl;
        }
        SatLit badlbl = satLit_Undef;
        {
            TimeKeeper tk(clausify_time_);
            Aig a = un_.unroll(target, k);
            badlbl = cnf.clausify(a);

            if (opts_.sat_preprocess) {
                TimeKeeper tk(preprocess_time_);
                solver->preprocess();
            }
        }

        assumptions[0] = badlbl;
        done = solver->solve(assumptions);

        if (!done) {
            if (cb_) {
                (*cb_)(k);
            }
            
            cls.clear();
            cls.push_back(~badlbl);
            solver->add_clause(cls);

            if (opts_.sat_preprocess) {
                solver->unfreeze_all();
            }
        } else {
            reached_depth_ = k;
        }
    }
            
    if (done) {
        for (size_t i = 0; i <= reached_depth_; ++i) {
            cex_trace_.push_back(AigList());
            AigList &c = cex_trace_.back();

            for (AigList::const_iterator it = model_->statevars().begin(),
                     end = model_->statevars().end(); it != end; ++it) {
                Aig a = *it;
                Aig ua = un_.unroll(a, i);
                SatLit l = cnf.lookup(ua);
                if (l != satLit_Undef) {
                    SatValue val = solver->get_model_value(l);
                    if (val != sat_Undef) {
                        c.push_back(aig_lit(a, val == sat_False));
                    }
                }
            }

            for (AigList::const_iterator it = model_->inputs().begin(),
                     end = model_->inputs().end(); it != end; ++it) {
                Aig a = *it;
                Aig ua = un_.unroll(a, i);
                SatLit l = cnf.lookup(ua);
                if (l != satLit_Undef) {
                    SatValue val = solver->get_model_value(l);
                    if (val != sat_Undef) {
                        c.push_back(aig_lit(a, val == sat_False));
                    }
                }
            }

            sort(c, aig_lt);
        }
    }

    delete solver;

    return done ? FALSIFIED : UNKNOWN;
}


bool BMC::get_counterexample_trace(std::vector<AigList> *out)
{
    if (cex_trace_.empty()) {
        return false;
    }

    *out = cex_trace_;

    return true;
}


Stats BMC::get_stats()
{
    Stats ret;
    std::ostringstream tmp;
    
#define ADDSTAT(name, val) do { tmp.str(""); \
        tmp << std::setprecision(3) << std::fixed << val;       \
        ret.push_back(std::make_pair(name, tmp.str()));         \
    } while (0)

    ADDSTAT("reached_depth", reached_depth_);
    ADDSTAT("clausify_time", clausify_time_);
    if (opts_.sat_preprocess) {
        ADDSTAT("preprocess_time", preprocess_time_);
    }
    ADDSTAT("prove_time", total_time_);
    size_t mem_used = get_mem_used_bytes() / (1024 * 1024);
    ADDSTAT("memory_used_mb", mem_used);

#undef ADDSTAT
    
    return ret;

}


bool BMC::check_reachable(const AigList &target)
{
    AigManager *mgr = model_->aig_manager();
    Aig t = mgr->aig_true();
    for (size_t i = 0; i < target.size(); ++i) {
        t = mgr->aig_and(t, target[i]);
    }
    Prover::Result res = do_prove(t);
    return (res == Prover::FALSIFIED);
}


} // namespace simplic3
