// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// CNF conversion routines
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "cnf.h"
#include "aig.hpp"
#include <assert.h>

namespace simplic3 {

//-----------------------------------------------------------------------------
// ClauseSink
//-----------------------------------------------------------------------------

bool CnfConv::ClauseSink::has_label(Aig a)
{
    assert(!AigManager::aig_is_negated(a));
    
    return parent_->cache_.find(a) != parent_->cache_.end();
}


void CnfConv::ClauseSink::set_label(Aig a)
{
    assert(!AigManager::aig_is_negated(a));

    SatVar v = parent_->dpll_->new_var();
    if (parent_->freeze_all_) {
        parent_->dpll_->set_frozen(v);
    }
    parent_->cache_[a] = v;
}


void CnfConv::ClauseSink::set_label(Aig a, int v)
{
    set_label(a);
}


void CnfConv::ClauseSink::begin_clause()
{
    parent_->clause_.clear();
}


void CnfConv::ClauseSink::add_literal(Aig a, bool negated)
{
    assert(has_label(a));

    SatVar v = parent_->cache_[a];
    parent_->clause_.push_back(SatLit(v, negated));
}


void CnfConv::ClauseSink::end_clause()
{
    parent_->dpll_->add_clause(parent_->clause_);
}


void CnfConv::ClauseSink::toplevel_literal(Aig a, bool negated)
{
    assert(has_label(a));

    SatVar v = parent_->cache_[a];
    toplit = SatLit(v, negated);
}


//-----------------------------------------------------------------------------
// CnfConv
//-----------------------------------------------------------------------------

CnfConv::CnfConv(Model *model, SatSolver *dpll, bool simple, bool freeze_all):
    model_(model),
    mgr_(model->aig_manager()),
    dpll_(dpll),
    simple_(simple),
    freeze_all_(freeze_all)
{
    SatVar v = dpll_->new_var();
    cache_[mgr_->aig_true()] = v;
    dpll_->add_clause(SatLit(v));
}


CnfConv::~CnfConv()
{
}


SatLit CnfConv::clausify(Aig a)
{
    SatLit ret = lookup(a);
    if (ret == satLit_Undef) {
        ClauseSink sink(this);
        if (simple_) {
            mgr_->aig2cnf(a, sink);
        } else {
            mgr_->aig2cnf_ls(a, sink);
        }
        ret = sink.toplit;
    }
    return ret;
}


SatLit CnfConv::lookup(Aig a)
{
    Aig aa = AigManager::aig_strip(a);
    
    CnfConvMap::iterator it = cache_.find(aa);
    if (it != cache_.end()) {
        bool neg = AigManager::aig_is_negated(a);
        SatVar v = it->second;
        return SatLit(v, neg);
    } else {
        return satLit_Undef;
    }
}


void CnfConv::set_label(Aig a, SatVar v)
{
    assert(cache_.find(a) == cache_.end());
    cache_[a] = v;
}


} // namespace simplic3
