// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// IC3 engine
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_IC3_H
#define SIMPLIC3_IC3_H

#include "simplic3/model.h"
#include "simplic3/unroll.h"
#include "simplic3/satsolver.h"
#include "simplic3/cnf.h"
#include "simplic3/miscutils.h"
#include "simplic3/opts.h"
#include "simplic3/ic3helpers.h"
#include "simplic3/prover.h"
#include <algorithm>

namespace simplic3 {

class IC3: public Prover {
public:
    IC3(Options opts, Model *model, size_t prop_idx);
    ~IC3();

    Result prove();
    Stats get_stats();

    void set_safe_bound_callback(const SafeBoundCallback *cb);
    
    bool get_counterexample_trace(std::vector<AigList> *out);
    bool get_final_invariant(std::vector<AigList> *out);

protected:
    friend class AbsIC3;
    
    void set_initial_clauses(const std::vector<AigList> &clauses);
    void initialize();
    void send_statevars(IC3Solver &s);
    void send_inputvars(IC3Solver &s);
    void send_init(IC3Solver &s, SatLit label);
    void send_trans(IC3Solver &s, SatLit label);
    Aig prop_aig();
    SatLit get_prop(IC3Solver &s, ModelUnroll &un, int k);
    void send_frame(IC3Solver &s, Frame &f, SatLit label);
    void reset_trans_solver(size_t i);
    IC3Solver &get_init_solver();
    IC3Solver &get_trans_solver(int time);
    IC3Solver &get_prop_solver();
    IC3Solver &get_trans_solver(int time, bool activate_frame);
    size_t get_trans_solver_idx(int time, bool offset=false);
    IC3Solver &get_prop_solver(bool activate_frame);
    
    virtual void new_frame();
    bool check_init();
    bool get_bad_cube(Cube *out);
    size_t depth();
    bool rec_block_cube(const Cube &c, int time);
    bool propagate_blocked_cubes();
    void extract_model(int time, Cube *out_state, Cube *out_inputs);
    void generalize_model(int time, const Cube &cur, const Cube &inputs,
                          const Cube *badstate, Cube *out);
    
    bool is_blocked(const Cube &c, int time);
    bool is_initial(const Cube &c);
    bool try_blocking(const Cube &c, int time, Cube *out, int *out_time,
                      bool compute_model, Cube *out_inputs=NULL);
    void generalize_and_push_forward(Cube &c, int &time);
    void generalize(Cube &c, int &time);
    void push_forward(Cube &c, int &time);
    void generalize_ctg(Cube &c, int &time, int rec_level);
    bool ctg_down(Cube &c, int &time, int rec_level);
    
    void add_blocked_cube(Cube &c, int time);
    void add_cube_as_clause(IC3Solver &s, const Cube &c);
    void add_cube_as_clause(IC3Solver &s, const Cube &c, SatLit lbl);
    void add_frame_clause(int time, FrameCube &pc);
    void gc_frame(int time);

    virtual void get_next(const Cube &c, Cube *out, int time);
    virtual bool is_cex(ProofObligationQueue &q);
    bool fixpoint(int time);
    
    bool solve(IC3Solver &s);

    SatLit aig2lit(IC3Solver &s, Aig a);

    void print_frames(int numcall);

    std::string cube2str(const Cube &c);

    static bool cube_lt(Aig a, Aig b)
    { return AigManager::aig_id(a) < AigManager::aig_id(b); }

    Options opts_;
    Model *model_;
    size_t prop_idx_;
    SatSolverFactory *factory_;

    std::vector<IC3Solver> solvers_;
    ModelUnroll un_;
    
    Random rng_;

    FrameList frames_;
    std::vector<size_t> frame_inactives_;
    bool use_gc_frame_;
    bool use_generalize_;

    SatLit prop_label_;
    SatLitList assumptions_;

    size_t rec_block_iter_;
    size_t solver_reset_frequency_;

    std::vector<Cube> cex_trace_;
    bool cex_trace_ok_;

    int verb_;

    typedef HashMap<Aig, float> ActMap;
    ActMap activity_;

    struct ActLt {
        ActMap &activity_;

        ActLt(ActMap &a): activity_(a) {}

        bool operator()(Aig a, Aig b) const
        {
            a = AigManager::aig_strip(a);
            b = AigManager::aig_strip(b);

            return activity_[a] < activity_[b];
        }
    };

    const SafeBoundCallback *cb_;

    std::vector<Cube> initial_cubes_;
    
    // statistics
    uint32_t num_solve_calls_;
    uint32_t num_solve_sat_calls_;
    uint32_t num_solve_unsat_calls_;
    uint32_t num_added_cubes_;
    uint32_t num_subsumed_cubes_;
    uint32_t num_reset_;
    uint32_t num_gc_frame_;
    uint32_t num_try_blocking_;
    uint32_t num_added_ctgs_;
    uint64_t num_solve_conflicts_;
    uint64_t num_solve_propagations_;
    uint64_t num_solve_decisions_;
    double solve_time_;
    double solve_sat_time_;
    double solve_unsat_time_;
    double rec_block_cube_time_;
    double generalize_model_time_;
    double generalize_and_push_time_;
    double propagate_forward_time_;
    double try_blocking_time_;
    double reset_time_;
    double clausify_time_;
    double preprocess_time_;
    double ctg_time_;
    double total_time_;
};


//-----------------------------------------------------------------------------
// Implementation of inline methods
//-----------------------------------------------------------------------------

inline SatLit IC3::aig2lit(IC3Solver &s, Aig a)
{
    SatLit ret = s.cnf->lookup(a);
    assert(ret != satLit_Undef);
    return ret;
}

inline size_t IC3::depth() { return frames_.size()-1; }

inline IC3Solver &IC3::get_init_solver()
{
    return solvers_[0];
}


inline size_t IC3::get_trans_solver_idx(int time, bool offset)
{
    size_t off = offset ? 1 : 0;
    if (opts_.max_trans_solvers <= 0) {
        return off + time;
    } else {
        int howmany = solvers_.size()-2;
        int limit = depth() - howmany;
        if (time <= limit) {
            return off;
        } else {
            return off + (time - limit);
        }
    }
}


inline IC3Solver &IC3::get_trans_solver(int time)
{
    size_t idx = get_trans_solver_idx(time, true);
    return solvers_[idx];
}


inline IC3Solver &IC3::get_trans_solver(int time, bool activate_frame)
{
    IC3Solver &s = get_trans_solver(time);
    for (size_t i = 0; i < frames_.size(); ++i) {
        if (i >= size_t(time) && activate_frame) {
            assumptions_.push_back(s.labels[i]);
        } else {
            assumptions_.push_back(~s.labels[i]);
        }
    }
    return s;
}


inline IC3Solver &IC3::get_prop_solver()
{
    return solvers_.back();
}


inline IC3Solver &IC3::get_prop_solver(bool activate_frame)
{
    IC3Solver &s = get_prop_solver();
    if (activate_frame) {
        assumptions_.push_back(s.labels[0]);
        assumptions_.push_back(~prop_label_);
    } else {
        assumptions_.push_back(~s.labels[0]);
        assumptions_.push_back(prop_label_);
    }
    return s;
}


} // namespace simplic3

#endif // SIMPLIC3_IC3_H
