// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Miscellaneous utility functions
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_MISCUTILS_H
#define SIMPLIC3_MISCUTILS_H

#include "simplic3/aig.h"
#include <assert.h>
#include <stdint.h>
#include <limits>
#include <algorithm>

namespace simplic3 {

class Random {
public:
    Random(uint32_t seed): seed_(seed) { assert(seed_); }
    ///< constructor. Precondition: seed must be different from zero

    uint32_t randint(uint32_t upper_bound);
    ///< returns a random integer 0 <= x < upper_bound

    double randfloat();
    ///< returns a random float 0 <= x < 1

private:
    uint64_t next() const;
    
    static const uint64_t a_ = 25214903917ULL;
    static const uint64_t c_ = 11U;
    static const uint64_t mask_ = ~(2ULL << 48);
    uint64_t seed_;
};


inline uint64_t Random::next() const
{
    return ((seed_ * a_) + c_) & mask_;
}


inline uint32_t Random::randint(uint32_t upper_bound)
{
    seed_ = next();
    return uint32_t(seed_ >> 16U) % upper_bound;
}


inline double Random::randfloat()
{
    uint32_t ub = std::numeric_limits<int>::max();
    return double(randint(ub)) / double(ub);
}


template <class T>
void random_shuffle(Random &rng, T &c)
{
    for (size_t i = 0; i < c.size(); ++i) {
        size_t idx = i + rng.randint(c.size()-i);
                
        assert(idx < c.size());

        std::swap(i, idx);
    }
}


double get_cpu_time_sec();
///< returns the total execution time

size_t get_mem_used_bytes();
///< returns the total amount of RAM used so far, including the
///< memory for loading the program and the needed shared libraries


class TimeKeeper {
public:
    inline TimeKeeper(double &out):
        out_(out)
    {
        reset();
    }
    
    inline ~TimeKeeper()
    {
        end_ = get_time_sec();
        double r = end_ - start_;
        out_ += r;
        out_ = std::max(out_, 0.0);
    }

    inline double get()
    {
        double e = get_time_sec();
        double r = e - start_;
        double ret = std::max(out_ + r, 0.0);
        return ret;
    }

    inline void reset()
    {
        start_ = get_time_sec();
        end_ = start_;
    }

private:
    inline double get_time_sec()
    {
        return get_cpu_time_sec();
    }
    
    double &out_;
    mutable double start_;
    mutable double end_;
};


template <class T, class Cmp>
void sort(T &v, Cmp c)
{
    std::sort(v.begin(), v.end(), c);
}

template <class T>
void sort(T &v)
{
    std::sort(v.begin(), v.end());
}


typedef std::vector< std::pair<std::string, std::string> > Stats;


inline bool aig_lt(Aig a, Aig b)
{ return AigManager::aig_id(a) < AigManager::aig_id(b); }

inline Aig aig_lit(Aig a, bool neg)
{ return neg ? AigManager::aig_not(a) : a; }

inline Aig aig_var(Aig a) { return AigManager::aig_strip(a); }
inline bool aig_sign(Aig a) { return AigManager::aig_is_negated(a); }


} // namespace simplic3

#endif // SIMPLIC3_MISCUTILS_H
