// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Model (finite state machine) representation
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_MODEL_H
#define SIMPLIC3_MODEL_H

#include "simplic3/aig.h"
extern "C" {
#include "simplic3/aiger.h"
}


namespace simplic3 {

typedef std::vector<Aig> AigList;
typedef HashSet<Aig> AigSet;


class Model {
public:
    Model();
    ~Model();
    
    bool init_from_aiger(aiger *aigmgr);
    bool init_from_aiger(const std::string &filename);

    const AigList &inputs() const;
    const AigList &properties() const;
    const AigList &statevars() const;
    
    bool is_statevar(Aig cur) const;
    bool is_inputvar(Aig e) const;
    bool is_nextstatevar(Aig e) const;

    Aig init_formula(Aig sv);
    Aig init_value(Aig sv) const;
    Aig next_value(Aig sv) const;
    Aig next(Aig sv) const;
    Aig cur(Aig nv) const;

    Aig newvar() { return mgr_.aig_var(nextvar_++); }
    int highest_var() { return nextvar_-1; }
    void set_highest_var(int v) { nextvar_ = v+1; }

    AigManager *aig_manager() { return &mgr_; }

    void add_aiger_input(Aig a);
    void add_aiger_output(Aig a);
    void add_aiger_latch(Aig v, Aig init, Aig next);

private:
    AigManager mgr_;

    struct Latch {
        Aig init;
        Aig trans;

        Latch(Aig i=AigManager::aig_null(), Aig t=AigManager::aig_null()):
            init(i), trans(t) {}
    };

    AigList inputs_;
    AigList properties_;
    AigList statevars_;
    typedef HashMap<Aig, Latch> LatchMap;
    LatchMap latchmap_;
    typedef HashMap<Aig, Aig> NextMap;
    NextMap nextmap_;
    NextMap curmap_;
    AigSet inputset_;
    
    int nextvar_;
};


} // namespace simplic3

#endif // SIMPLIC3_MODEL_H
