// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Options handling
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "opts.h"
#include "satsolver.h"
#include "prover.h"
#include <stdlib.h>
#include <algorithm>
#include <sstream>

namespace simplic3 {

Options::Options()
{
    algorithm = "ic3";
    solver = "minisat";
    use_relind = true;
    short_cex = false;
    verbosity = 0;
    prop_unroll = 4;
    solver_reset_freq = -1;
    max_trans_solvers = 25;
    sat_preprocess = true;
    cnf_simple = false;
    use_ctg = false;
    bmc_mindepth = 0;
    bmc_maxdepth = 0;
    aic3_ref_bmc = false;
    simpl_max_td_depth = 0;
    simpl_max_sim_depth = 200;
    simpl_min_td_depth = 0;
}


namespace {

int getopt(const char *name, const std::string &val)
{
    if (val.empty()) {
        throw Options::error(
            std::string("missing argument for option") + name);
    } else {
        return atoi(val.c_str());
    }
}

} // namespace


bool Options::set(const std::string &o, const std::string &val)
{
    if (o == "-v") {
        verbosity = getopt("-v", val);
    } else if (o == "-a") {
        algorithm = val;
        std::vector<std::string> names = Prover::getnames();
        if (std::find(names.begin(), names.end(), algorithm) ==
            names.end()) {
            std::string msg("unknown algorithm `");
            msg += algorithm + "'";
            throw error(msg);
        }
    } else if (o == "-s") {
        solver = val;
        std::vector<std::string> names = SatSolverFactory::getnames();
        if (std::find(names.begin(), names.end(), solver) ==
            names.end()) {
            std::string msg("unknown solver `");
            msg += solver + "'";
            throw error(msg);
        }
    } else if (o == "-i") {
        use_relind = getopt("-i", val);
    } else if (o == "-x") {
        short_cex = getopt("-x", val);
    } else if (o == "-u") {
        prop_unroll = getopt("-u", val);
    } else if (o == "-r") {
        solver_reset_freq = getopt("-r", val);
    } else if (o == "-m") {
        max_trans_solvers = getopt("-m", val);
    } else if (o == "-p") {
        sat_preprocess = getopt("-p", val);
    } else if (o == "-c") {
        cnf_simple = !getopt("-c", val);
    } else if (o == "-g") {
        use_ctg = getopt("-g", val);
    } else if (o == "-b") {
        bmc_mindepth = getopt("-b", val);
    } else if (o == "-k") {
        bmc_maxdepth = getopt("-k", val);
    } else if (o == "-z") {
        aic3_ref_bmc = getopt("-z", val);
    } else if (o == "-t") {
        simpl_max_td_depth = getopt("-t", val);
    } else if (o == "-d") {
        simpl_min_td_depth = getopt("-d", val);
    } else {
        return false;
    }

    return true;
}


std::vector<std::string> Options::parse(const std::vector<std::string> &args)
{
    std::vector<std::string> ret;
    
    for (size_t i = 0; i < args.size(); ++i) {
        const std::string &o = args[i];
        if (o == "-h") {
            set(o, "");
        } else if (!set(o, i+1 < args.size() ? args[i+1] : "")) {
            ret.push_back(o);
        } else {
            ++i;
        }
    }

    return ret;
}


std::vector<std::string> Options::parse(int argc, const char **argv)
{
    std::vector<std::string> args;
    for (int i = 1; i < argc; ++i) {
        args.push_back(argv[i]);
    }
    return parse(args);
}


void Options::print_help(std::ostream &out, const std::string &msg)
{
    if (!msg.empty()) {
        out << msg << "\n";
    }
    std::vector<std::string> names;
    out << "  -v N : set verbosity level\n"
        << "  -a ALGORITHM : select algorithm to use. ";
    names = Prover::getnames();
    out << "Available algorithms:\n";
    for (size_t i = 0; i < names.size(); ++i) {
        out << "                  " << names[i] << "\n";
    }
    out << "  -s SOLVER : select SAT solver to use. ";
    names = SatSolverFactory::getnames();
    out << "Available solvers:\n";
    for (size_t i = 0; i < names.size(); ++i) {
        out << "               " << names[i] << "\n";
    }
    out << "  -i B : if B is zero, do not use relative induction\n"
        << "  -g B : if B is nonzero, use CTG-based cube generalization\n"
        << "  -x B : if B is nonzero, generate shortest counterexamples\n"
        << "  -u K : unroll property for K cycles\n"
        << "  -r N : reset sat solver every N temporary clauses "
        << "(0: never reset)\n"
        << "  -m N : max number of solvers to use\n"
        << "  -p B : if B is nonzero, use SAT preprocessing (if the backend supports it)\n"
        << "  -c B : select CNF conversion algorithm to use (0: simple, 1: advanced)\n"
        << "  -b K : starting bound for BMC-based algorithms\n"
        << "  -k K : maximum bound for BMC-based algorithms (0: unbounded)\n"
        << "  -z B : if nonzero, use BMC for abstraction refinement in aic3\n"
        << "  -t N : if nonzero, simplify model using temporal decomposition up to depth N\n"
        << "  -d N : if nonzero, apply temporal decomposition of N steps even when no simplifications are detected\n";
    
    out.flush();
}


namespace {

template <class T>
std::string stropt(T val)
{
    std::ostringstream tmp;
    tmp << val;
    return tmp.str();
}

} // namespace

std::string Options::get(const std::string &o)
{
    if (o == "-v") {
        return stropt(verbosity);
    } else if (o == "-s") {
        return solver;
    } else if (o == "-i") {
        return stropt(use_relind);
    } else if (o == "-x") {
        return stropt(short_cex);
    } else if (o == "-u") {
        return stropt(prop_unroll);
    } else if (o == "-r") {
        return stropt(solver_reset_freq);
    } else if (o == "-m") {
        return stropt(max_trans_solvers);
    } else if (o == "-p") {
        return stropt(sat_preprocess);
    } else if (o == "-c") {
        return stropt(cnf_simple);
    } else if (o == "-g") {
        return stropt(use_ctg);
    } else if (o == "-b") {
        return stropt(bmc_mindepth);
    } else if (o == "-k") {
        return stropt(bmc_maxdepth);
    } else if (o == "-z") {
        return stropt(aic3_ref_bmc);
    } else if (o == "-t") {
        return stropt(simpl_max_td_depth);
    } else if (o == "-d") {
        return stropt(simpl_min_td_depth);
    } else {
        throw error(std::string("unknown option `") + o + "'");
    }
}

} // namespace simplic3
