// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Picosat SAT solver backend
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "picosatsolver.h"
#include <assert.h>

namespace simplic3 {

//-----------------------------------------------------------------------------
// PicosatSolver
//-----------------------------------------------------------------------------

PicosatSolver::PicosatSolver()
{
    dpll_ = picosat_init();
    
    gclabel_ = satLit_Undef;
}


PicosatSolver::~PicosatSolver()
{
    picosat_reset(dpll_);
}


namespace {

inline SatLit lit2satlit(int l) { return SatLit(l < 0 ? -l : l, l < 0); }
inline int satlit2lit(SatLit l) { return sign(l) ? -var(l) : var(l); }

inline SatValue lbool2val(int l)
{
    if (l == 1) {
        return sat_True;
    } else if (l == -1) {
        return sat_False;
    } else {
        return sat_Undef;
    }
}

inline int val2lbool(SatValue l)
{
    if (l == sat_True) {
        return 1;
    } else if (l == sat_False) {
        return -1;
    } else {
        return 0;
    }
}

} // namespace


SatVar PicosatSolver::new_var()
{
    int v = picosat_inc_max_var(dpll_);
    return v;
}


void PicosatSolver::add_clause(const SatLitList &cls, SatLit label)
{
    for (size_t i = 0; i < cls.size(); ++i) {
        picosat_add(dpll_, satlit2lit(cls[i]));
    }
    if (label != satLit_Undef) {
        picosat_add(dpll_, satlit2lit(label));
    }
    picosat_add(dpll_, 0);
}


void PicosatSolver::add_clause(const SatLitList &cls)
{
    add_clause(cls, satLit_Undef);
}


bool PicosatSolver::solve(const SatLitList &assumps)
{
    assumptions_.clear();
    for (size_t i = 0; i < assumps.size(); ++i) {
        int l = satlit2lit(assumps[i]);
        assumptions_.push_back(l);
        picosat_assume(dpll_, l);
    }
    int res = picosat_sat(dpll_, -1);
    return res == PICOSAT_SATISFIABLE;
}


SatValue PicosatSolver::get_model_value(SatLit l)
{
    int val = picosat_deref(dpll_, satlit2lit(l));
    return lbool2val(val);
}


void PicosatSolver::get_unsat_core(SatLitSet &out)
{
    for (size_t i = 0; i < assumptions_.size(); ++i) {
        int l = assumptions_[i];
        if (picosat_failed_assumption(dpll_, l)) {
            out.insert(lit2satlit(-l));
        }
    }
}


void PicosatSolver::push()
{
    picosat_push(dpll_);
}


void PicosatSolver::pop()
{
    picosat_pop(dpll_);
}


//-----------------------------------------------------------------------------
// PicosatSolverFactory
//-----------------------------------------------------------------------------

SatSolver *PicosatSolverFactory::new_satsolver()
{
    return new PicosatSolver();
}


} // namespace simplic3
