// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Generic SAT solver interface
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_SATSOLVER_H
#define SIMPLIC3_SATSOLVER_H

#include "simplic3/hashmap.h"
#include <stdint.h>
#include <vector>

namespace simplic3 {

enum SatValue {
    sat_Undef, 
    sat_False,
    sat_True
};

typedef int SatVar;
const SatVar satVar_Undef = 0;

class SatLit {
public:
    explicit SatLit(SatVar v=satVar_Undef, bool neg=false):
        repr_(neg ? -v : v) {}
    friend SatVar var(SatLit l);
    friend bool sign(SatLit l);
    friend SatLit operator~(SatLit l);
    size_t hash() const { return repr_; }
    bool operator==(SatLit o) const { return repr_ == o.repr_; }
    bool operator!=(SatLit o) const { return repr_ != o.repr_; }

private:
    int repr_;
};

const SatLit satLit_Undef;

inline SatVar var(SatLit l) { return sign(l) ? -l.repr_ : l.repr_; }
inline bool sign(SatLit l) { return l.repr_ < 0; }
inline SatLit operator~(SatLit l) { l.repr_ *= -1; return l; }

typedef std::vector<SatLit> SatLitList;
typedef HashSet<SatLit, Hasher<SatLit> > SatLitSet;

class SatSolver {
public:
    virtual ~SatSolver() {}
    virtual SatVar new_var() = 0;
    virtual void add_clause(const SatLitList &cls) = 0;
    virtual void add_clause(SatLit unit);
    virtual void add_clause(SatLit l1, SatLit l2);
    virtual bool solve(const SatLitList &assumps) = 0;
    virtual SatValue get_model_value(SatLit l) = 0;
    virtual void get_unsat_core(SatLitSet &out) = 0;

    virtual void push() = 0;
    virtual void pop() = 0;

    virtual void set_frozen(SatVar v) = 0;
    virtual void unfreeze_all() {}
    virtual void preprocess() {}

    virtual int reset_frequency() const { return 0; }

    virtual uint64_t get_last_propagations() const { return 0; }
    virtual uint64_t get_last_conflicts() const { return 0; }
    virtual uint64_t get_last_decisions() const { return 0; }
};

inline void SatSolver::add_clause(SatLit unit)
{ SatLitList tmp(1, unit); add_clause(tmp); }

inline void SatSolver::add_clause(SatLit l1, SatLit l2)
{ SatLitList tmp(2, l1); tmp[1] = l2; add_clause(tmp); }


class SatSolverFactory {
public:
    virtual ~SatSolverFactory();
    virtual SatSolver *new_satsolver() = 0;

    static SatSolverFactory *get(const std::string &name);
    static std::vector<std::string> getnames();
};

} // namespace simplic3

#endif // SIMPLIC3_SATSOLVER_H
