// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Model unrolling
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "unroll.h"
#include <assert.h>
#include <algorithm>

namespace simplic3 {

ModelUnroll::ModelUnroll(Model *model, bool internal_mgr):
    model_(model),
    mgr_(),
    internal_mgr_(internal_mgr)
{
    nextvar_ = model_->highest_var()+1;
}


Aig ModelUnroll::unroll(Aig a, int depth)
{
    assert(depth >= 0);

    AigManager *mgr = aig_manager();
    
    std::vector<Key> to_process;
    Aig true_aig = model_->aig_manager()->aig_true();

    to_process.push_back(Key(AigManager::aig_strip(a), depth));
    while (!to_process.empty()) {
        Aig cur = to_process.back().a;
        int k = to_process.back().k;

        Key entry(cur, k);
        if (cache_.find(entry) != cache_.end()) {
            to_process.pop_back();
            continue;
        }

        Aig res = mgr->aig_null();
        if (model_->is_statevar(cur)) {
            if (k == 0) {
                res = mgr->aig_var(model_->aig_manager()->aig_get_var(cur));
                to_process.pop_back();
                cache_[entry] = res;
            } else {
                Aig n = model_->next_value(cur);
                bool neg = AigManager::aig_is_negated(n);
                n = AigManager::aig_strip(n);
                Key n_entry(n, k-1);
                UnrollMap::iterator it = cache_.find(n_entry);
                if (it == cache_.end()) {
                    to_process.push_back(n_entry);
                } else {
                    res = it->second;
                    if (neg) {
                        res = mgr->aig_not(res);
                    }
                    to_process.pop_back();
                    cache_[entry] = res;
                }
            }
        } else if (model_->is_inputvar(cur)) {
            to_process.pop_back();
            if (k == 0) {
                res = mgr->aig_var(model_->aig_manager()->aig_get_var(cur));
            } else {
                if (internal_mgr_) {
                    res = mgr->aig_var(nextvar_++);
                } else {
                    res = model_->newvar();
                }
            }
            cache_[entry] = res;
        } else if (cur == true_aig) {
            to_process.pop_back();
            res = mgr->aig_true();
            cache_[entry] = res;
        } else {
            bool children_done = true;

            assert(AigManager::aig_is_and(cur));
            
            Aig l = AigManager::aig_get_left(cur);
            Aig r = AigManager::aig_get_right(cur);

            bool neg_l = AigManager::aig_is_negated(l);
            bool neg_r = AigManager::aig_is_negated(r);

            l = AigManager::aig_strip(l);
            r = AigManager::aig_strip(r);

            Key entry_l(l, k);
            Key entry_r(r, k);

            if (cache_.find(entry_r) == cache_.end()) {
                children_done = false;
                to_process.push_back(entry_r);
            }
            if (cache_.find(entry_l) == cache_.end()) {
                children_done = false;
                to_process.push_back(entry_l);
            }

            if (children_done) {
                to_process.pop_back();
                
                Aig rl = cache_[entry_l];
                Aig rr = cache_[entry_r];

                if (neg_l) {
                    rl = mgr->aig_not(rl);
                }
                if (neg_r) {
                    rr = mgr->aig_not(rr);
                }

                res = mgr->aig_and(rl, rr);
                cache_[entry] = res;
            }
        }
    }

    Aig ret = cache_[Key(AigManager::aig_strip(a), depth)];
    if (AigManager::aig_is_negated(a)) {
        ret = mgr->aig_not(ret);
    }

    return ret;
}


} // namespace simplic3
