#!/usr/bin/env python

# SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
# for finite-state functional transition systems
# 
# A simple script to run a portfolio of SimplIC3 instances with different
# algorithms.
# 
# Author: Alberto Griggio <griggio@fbk.eu>
# See LICENSE.txt for copyright/licensing information
# See CREDITS.txt for other credits
 
from simplic3 import *
import multiprocessing
import sys
import signal
import optparse
import time


def run_prover(name, solvers, idx, q, hwmcc, stdout_lock):
    def cb(k):
        stdout_lock.acquire()
        try:
            sys.stdout.write('u%d\n' % k)
            sys.stdout.flush()
        finally:
            stdout_lock.release()

    result, wit, stats = 2, None, ""
    
    for s in solvers:
        if hwmcc:
            err = simplic3_set_safe_bound_callback(s, cb)
            if err:
                raise Exception("error setting callback function")
        result = simplic3_prove(s, idx)
        wit = None
        if result == 1:
            wit = simplic3_witness(s)
        stats = simplic3_stats(s)
        if result in (0, 1):
            break
        
    q.put((name, result, wit, stats))


def Prover(*args): return tuple(args)

_prover_opts = {
    'ic3': Prover([('-a', 'ic3')]),
    'ic3g': Prover([('-a', 'ic3'), ('-g', '1')]),
    'aic3': Prover([('-a', 'aic3'), ('-z', '1')]),
    'bmc': Prover([('-a', 'bmc')]),
    'ind': Prover([('-a', 'ind')]),
    'indbmc': Prover([('-a', 'ind'), ('-k', '45')],
                     [('-a', 'bmc'), ('-b', '46')]),
    'ic3t': Prover([('-a', 'ic3'), ('-t', '32')]),
    }

_all_provers = sorted(_prover_opts.keys())

_prover_configs = {
    'default' : ['ic3', 'ic3g', 'aic3', 'indbmc'],
    'simple' : ['ic3', 'bmc'],
    'experimental' : ['ic3t', 'ic3g', 'aic3', 'indbmc'],
    }


def create_prover(name, aigmgr, idx, verb, q, hwmcc, stdout_lock):
    solvers = []
    for cfg in _prover_opts[name]:
        s = simplic3_new()
        try:
            for (k, v) in cfg: #_prover_opts[name]:
                err = simplic3_setopt(s, k, v)
                if err:
                    raise Exception("error setting %s %s" % (k, v))
        except KeyError:
            raise Exception("unknown prover %s" % name)

        if verb:
            simplic3_setopt(s, "-v", "1")
        err = simplic3_init(s, aigmgr)
        if err:
            raise Exception("error initializing %s" % name)
        solvers.append(s)
    p = multiprocessing.Process(target=run_prover,
                                args=(name, solvers, idx, q, hwmcc,
                                      stdout_lock))
    return p


def parse_cmdline():
    p = optparse.OptionParser('%prog [options] FILENAME.aig')
    p.add_option('-p', '--prover', action='append',
                 help='activate the given prover. Available provers are: %s' % \
                 " ".join(_all_provers))
    p.add_option('-c', '--config',
                 help='activate the given configuration (set of provers). ' \
                 'Takes precedence over -p. If no configuration ' \
                 'and no prover is given, the default configuration ' \
                 'will be activated. Available configurations are: %s' % \
                 " ".join(sorted(_prover_configs.keys())))
    p.add_option('-v', '--verbose', action='store_true',
                 help='verbose output')
    p.add_option('-s', '--stats', action='store_true',
                 help='print stats at the end')
    p.add_option('-i', '--index', type='int',
                 help='index of the property to prove (0 by default)')
    p.add_option('--hwmcc', action='store_true',
                 help='use output format of HWMCC.')
    
    opts, args = p.parse_args()
    if opts.config:
        try:
            opts.prover = _prover_configs[opts.config.lower()]
        except KeyError:
            raise Exception("unknown configuration %s" % opts.config)
    elif not opts.prover:
        opts.config = 'default'
        opts.prover = _prover_configs['default']
    ## if not opts.prover:
    ##     opts.prover = _all_provers
    if not opts.index:
        opts.index = 0

    filename = None
    if not args:
        filename = "/dev/stdin"
    elif len(args) > 1:
        raise Exception("multiple non-option arguments given")
    else:
        filename = args[0]

    return opts, filename


def print_hwmcc_witness(aigmgr, res, wit):
    pr = sys.stdout.write
    pr("%d\n" % res)
    if res == 1 and wit:
        inps = []
        for i in range(aiger_num_inputs(aigmgr)):
            a = aiger_get_input(aigmgr, i)
            inps.append(aiger_lit2var(a.lit))

        for sl in wit:
            m = {}
            for l in sl:
                m[aiger_lit2var(l)] = aiger_sign(l) and '0' or '1'
            out = []
            for v in inps:
                out.append(m.get(v, 'x'))
            pr("%s\n" % "".join(out))


def main():
    start = time.time()
    opts, filename = parse_cmdline()
        
    aigmgr = aiger_init()
    err = aiger_open_and_read_from_file(aigmgr, filename);
    if err:
        raise Exception(err)
    
    q = multiprocessing.Queue(len(_all_provers))
    stdout_lock = multiprocessing.Lock()

    if opts.stats and opts.config:
        if opts.hwmcc:
            pr = sys.stderr.write
        else:
            pr = sys.stdout.write
        pr('activating configuration %s\n' % opts.config)

    provers = []
    for name in sorted(set(opts.prover)):
        p = create_prover(name, aigmgr, opts.index, opts.verbose, q,
                          opts.hwmcc, stdout_lock)
        provers.append(p)

    def exit_gracefully(signum, frame):
        sys.stderr.write('Interrupted by signal %d\nUnknown\n' % signum)
        for p in provers:
            if p.is_alive():
                p.terminate()
        exit(2)

    signal.signal(signal.SIGXCPU, exit_gracefully)
    signal.signal(signal.SIGINT, exit_gracefully)
    
    for p in provers:
        p.start()

    n = len(provers)
    safe_bound = -1
    while n > 0:
        data = q.get()
        if False: #isinstance(data, int):
            if data > safe_bound:
                sys.stdout.write('u%d\n' % data)
                sys.stdout.flush()
                safe_bound = data
        else:
            name, res, wit, stats = data #q.get()
            n -= 1
            if res in (0, 1):
                break
        
    for p in provers:
        if p.is_alive():
            p.terminate()

    if opts.hwmcc:
        pr = sys.stderr.write
    else:
        pr = sys.stdout.write

    if opts.stats:
        if res in (0, 1):
            pr('Solution found by %s\n' % name)
        else:
            pr('No solution found\n')
        for (k, v) in stats:
            pr("%s = %s\n" % (k, v))
        end = time.time()
        pr("total_time = %.3f\n" % (end - start))

    if opts.stats or not opts.hwmcc:
        pr("%s\n" % {0 : 'Safe', 1 : 'Unsafe'}.get(res, 'Unknown'))

    if opts.hwmcc:
        print_hwmcc_witness(aigmgr, res, wit)

    aiger_reset(aigmgr)

    exit(res)


if __name__ == '__main__':
    try:
        main()
    except Exception, e:
        pr("ERROR: %s\n" % e)
        pr('Unknown\n')
        exit(2)
    
