/* -*- fundamental -*- */
/* SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
 * for finite-state functional transition systems
 *  
 * SWIG interface file to create the Python API for SimplIC3
 * 
 * Author: Alberto Griggio <griggio@fbk.eu>
 * See LICENSE.txt for copyright/licensing information
 * See CREDITS.txt for other credits
 */

%include "typemaps.i"

/* custom typemaps for return values that are dynamically allocated */

%typemap(out) char *simplic3_opthelp(SimplIC3 s) {
  if ($1 == NULL) {
     Py_INCREF(Py_None);
     $result = Py_None;
  } else {
     $result = PyString_FromString($1);
     simplic3_free($1);
  }
}

%typemap(out) unsigned int *simplic3_witness(SimplIC3 s) {
  if ($1 == NULL) {
     Py_INCREF(Py_None);
     $result = Py_None;
  } else {
     size_t i = 0;
     PyObject *cur = NULL;
     unsigned int *w = $1;

     $result = PyList_New(0);
     cur = PyList_New(0);
     
     while (1) {
         if (!w[i++]) {
             PyList_Append($result, cur);
             if (!w[i]) {
                 break;
             } else {
                 cur = PyList_New(0);
             }
         } else {
             PyList_Append(cur, PyInt_FromSize_t(w[i-1]));
         }
     }
     
     simplic3_free(w);
  }
}

%typemap(out) char **simplic3_stats(SimplIC3 s) {
   if ($1 == NULL) {
      Py_INCREF(Py_None);
      $result = Py_None;
   } else {
      size_t i = 0;
      PyObject *cur = NULL;
      char **s = $1;
      
      $result = PyList_New(0);

      while (1) {
          if (!s[i]) {
              break;
          } else {
              char *k = s[i++];
              char *v = s[i++];
              
              cur = PyTuple_Pack(2, PyString_FromString(k),
                                    PyString_FromString(v));

              simplic3_free(v);
              simplic3_free(k);

              PyList_Append($result, cur);
          }
      }

      simplic3_free(s);
   }
}


%typemap(in) (void(*cb)(int, void *), void *user_data) {
    if ($input == Py_None) {
        $1 = NULL;
        $2 = NULL;
    } else {
        if (!PyCallable_Check($input)) {
            PyErr_SetString(PyExc_TypeError, "Callable object required!\n");
            return NULL;
        }
        $1 = simplic3_call_python_callable;
        $2 = (void *)$input;
    }
}

%exception simplic3_set_safe_bound_callback {
    if (setjmp(exception_buffer) == 0) {
        $action
    } else {
        SWIG_fail;
    }
}


%ignore simplic3_free;
%ignore aiger_read_from_string;

/*****************************************************************************/

%module simplic3
%{
#include "simplic3.h"
#include <setjmp.h>
%}

%include "simplic3/aiger.h"
%include "simplic3.h"

unsigned aiger_num_inputs(aiger *a);
unsigned aiger_num_latches(aiger *a);
unsigned aiger_num_outputs(aiger *a);
unsigned aiger_num_ands(aiger *a);
unsigned aiger_num_bad(aiger *a);
unsigned aiger_num_constraints(aiger *a);
unsigned aiger_num_justice(aiger *a);
unsigned aiger_num_fairness(aiger *a);

aiger_symbol aiger_get_input(aiger *a, unsigned i);
aiger_symbol aiger_get_latch(aiger *a, unsigned i);
aiger_symbol aiger_get_output(aiger *a, unsigned i);
aiger_symbol aiger_get_bad(aiger *a, unsigned i);
aiger_symbol aiger_get_constraint(aiger *a, unsigned i);
aiger_symbol aiger_get_justice(aiger *a, unsigned i);
aiger_symbol aiger_get_fairness(aiger *a, unsigned i);
aiger_and aiger_get_and(aiger *a, unsigned i);

unsigned aiger_sign(unsigned l);
unsigned aiger_strip(unsigned l);
unsigned aiger_not(unsigned l);
unsigned aiger_var2lit(unsigned i);
unsigned aiger_lit2var(unsigned i);

%{

unsigned aiger_num_inputs(aiger *a) { return a->num_inputs; }
unsigned aiger_num_latches(aiger *a) { return a->num_latches; }
unsigned aiger_num_outputs(aiger *a) { return a->num_outputs; }
unsigned aiger_num_ands(aiger *a) { return a->num_ands; }
unsigned aiger_num_bad(aiger *a) { return a->num_bad; }
unsigned aiger_num_constraints(aiger *a) { return a->num_constraints; }
unsigned aiger_num_justice(aiger *a) { return a->num_justice; }
unsigned aiger_num_fairness(aiger *a) { return a->num_fairness; }

aiger_symbol aiger_get_input(aiger *a, unsigned i) { return a->inputs[i]; }
aiger_symbol aiger_get_latch(aiger *a, unsigned i) { return a->latches[i]; }
aiger_symbol aiger_get_output(aiger *a, unsigned i) { return a->outputs[i]; }
aiger_symbol aiger_get_bad(aiger *a, unsigned i) { return a->bad[i]; }
aiger_symbol aiger_get_constraint(aiger *a, unsigned i) { return a->constraints[i]; }
aiger_symbol aiger_get_justice(aiger *a, unsigned i) { return a->justice[i]; }
aiger_symbol aiger_get_fairness(aiger *a, unsigned i) { return a->fairness[i];}

aiger_and aiger_get_and(aiger *a, unsigned i) { return a->ands[i]; }

unsigned aiger_sign(unsigned l) { return (((unsigned)(l))&1); }
unsigned aiger_strip(unsigned l) { return (((unsigned)(l))&~1); }
unsigned aiger_not(unsigned l) { return (((unsigned)(l))^1); }
unsigned aiger_var2lit(unsigned i) { return (((unsigned)(i)) << 1); }
unsigned aiger_lit2var(unsigned l) { return (((unsigned)(l)) >> 1); }


static jmp_buf exception_buffer;

static void simplic3_call_python_callable(int k, void *data)
{
    PyObject *callable = (PyObject *)data;
    PyObject *result;
    PyObject *args;
    int retval;
    /* call the python callback */
    args = Py_BuildValue("(i)", k);
    result = PyObject_CallObject(callable, args);
    if (PyErr_Occurred()) {
        longjmp(exception_buffer, 1);
    }
    Py_XDECREF(result);
    /* free the args list */
    Py_DECREF(args);
}


%}
