// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// IC3 engine
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "ic3.h"
#include "modelsim.h"
#include <math.h>
#include <assert.h>
#include <list>
#include <string.h>
#include <iomanip>
#include <sstream>
#include <memory>
#include <iostream>

namespace simplic3 {

//-----------------------------------------------------------------------------
// IC3
//-----------------------------------------------------------------------------

IC3::IC3(Options opts, Model *model, size_t prop_idx):
    opts_(opts),
    model_(model),
    prop_idx_(prop_idx),
    rng_(42),
    un_(model_),
    cb_(NULL)
{
    factory_ = NULL;
    cex_trace_ok_ = false;
    use_gc_frame_ = true;
    use_generalize_ = true;

    num_solve_calls_ = 0;
    num_solve_sat_calls_ = 0;
    num_solve_unsat_calls_ = 0;
    num_added_cubes_ = 0;
    num_subsumed_cubes_ = 0;
    num_reset_ = 0;
    num_gc_frame_ = 0;
    num_try_blocking_ = 0;
    num_added_ctgs_ = 0;
    num_solve_conflicts_ = 0;
    num_solve_decisions_ = 0;
    num_solve_propagations_ = 0;
    
    solve_time_ = 0.0;
    solve_sat_time_ = 0.0;
    solve_unsat_time_ = 0.0;
    rec_block_cube_time_ = 0.0;
    generalize_model_time_ = 0.0;
    generalize_and_push_time_ = 0.0;
    propagate_forward_time_ = 0.0;
    try_blocking_time_ = 0.0;
    reset_time_ = 0.0;
    clausify_time_ = 0.0;
    preprocess_time_ = 0.0;
    ctg_time_ = 0.0;
    total_time_ = 0.0;

    rec_block_iter_ = 0;
    solver_reset_frequency_ = 0;

    verb_ = opts_.verbosity;
}


IC3::~IC3()
{
    for (size_t i = 0; i < solvers_.size(); ++i) {
        solvers_[i].destroy();
    }
}


void IC3::set_safe_bound_callback(const SafeBoundCallback *cb)
{
    cb_ = cb;
}


void IC3::set_initial_clauses(const std::vector<AigList> &clauses)
{
    for (size_t i = 0; i < clauses.size(); ++i) {
        initial_cubes_.push_back(Cube());
        Cube &c = initial_cubes_.back();
        const AigList &cl = clauses[i];
        c.reserve(cl.size());
        for (size_t j = 0; j < cl.size(); ++j) {
            c.push_back(model_->aig_manager()->aig_not(cl[j]));
        }
    }
}


Prover::Result IC3::prove()
{
    TimeKeeper t(total_time_);
    initialize();

    new_frame();
    if (!check_init()) {
        if (verb_ >= 2) {
            std::cout << "property violated by initial states" << std::endl;
        }
        return FALSIFIED;
    } else if (cb_) {
        (*cb_)(opts_.prop_unroll);
    }
    new_frame();
    for (size_t i = 0; i < initial_cubes_.size(); ++i) {
        FrameCube pc(initial_cubes_[i]);
        add_frame_clause(depth(), pc);
    }

    Prover::Result res = UNKNOWN;
    int max_k = opts_.bmc_maxdepth;
    
    while (true) {
        if (max_k && depth() > max_k) {
            break;
        }
        
        Cube c;
        if (get_bad_cube(&c)) {
            if (verb_ >= 2) {
                std::cout << "got bad cube of size " << c.size()
                          << " at depth " << depth() << std::endl;
            }
            if (!rec_block_cube(c, depth())) {
                res = FALSIFIED;
                break;
            }
        } else {
            if (verb_ >= 2) {
                std::cout << "current frame satisfies the property, "
                          << "propagate forward" << std::endl;
            }
            rec_block_iter_ = 0;
            if (cb_) {
                (*cb_)(depth() + opts_.prop_unroll);
            }
            new_frame();
            bool fix = propagate_blocked_cubes();
            print_frames(frames_.size()-1);
            if (fix) {
                res = VERIFIED;
                break;
            }
        }
    }

    return res;
}


void IC3::initialize()
{
    factory_ = SatSolverFactory::get(opts_.solver);

    solvers_.push_back(IC3Solver());
    IC3Solver &init_solver = solvers_.back();
    init_solver.init(model_, factory_, opts_.cnf_simple);
    send_statevars(init_solver);
    send_init(init_solver, satLit_Undef);

    solvers_.push_back(IC3Solver());
    IC3Solver &prop_solver = solvers_.back();
    prop_solver.init(model_, factory_, opts_.cnf_simple);
    send_statevars(prop_solver);
    send_inputvars(prop_solver);
    prop_label_ = get_prop(prop_solver, un_, opts_.prop_unroll);
    prop_solver.addlabel();
    
    solver_reset_frequency_ = prop_solver.solver->reset_frequency();
    if (opts_.solver_reset_freq >= 0) {
        solver_reset_frequency_ = opts_.solver_reset_freq;
    }

    size_t nvars = model_->statevars().size();
    size_t i = 0;
    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it, ++i) {
        Aig v = *it;
        activity_[v] = (nvars - i - 1) / (float)nvars;
    }

    cex_trace_ok_ = false;
}


void IC3::send_init(IC3Solver &solver, SatLit label)
{
    TimeKeeper tk(clausify_time_);
    
    SatSolver *dpll = solver.solver;
    CnfConv *cnf = solver.cnf;

    if (label != satLit_Undef) {
        dpll->set_frozen(var(label));
    }
    
    SatLitList cls;

    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it) {
        Aig v = *it;
        Aig a = model_->init_formula(v);
        SatLit l = cnf->clausify(a);

        cls.clear();
        if (label != satLit_Undef) {
            cls.push_back(~label);
        }
        cls.push_back(l);
        dpll->add_clause(cls);
    }
}


void IC3::send_trans(IC3Solver &solver, SatLit label)
{
    TimeKeeper tk(clausify_time_);

    SatSolver *dpll = solver.solver;
    CnfConv *cnf = solver.cnf;

    if (label != satLit_Undef) {
        dpll->set_frozen(var(label));
    }
    
    SatLitList cls;

    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it) {
        Aig v = *it;
        Aig a = model_->next_value(v);
        Aig n = model_->next(v);
        SatLit l1 = cnf->clausify(a);
        dpll->set_frozen(var(l1));
    }
}


inline Aig IC3::prop_aig()
{
    return model_->properties()[prop_idx_];
}


SatLit IC3::get_prop(IC3Solver &solver, ModelUnroll &un, int k)
{
    TimeKeeper tk(clausify_time_);

    SatSolver *dpll = solver.solver;
    CnfConv *cnf = solver.cnf;

    Aig a = prop_aig();
    if (k > 0) {
        a = un.unroll(a, k);
    }
    SatLit l = cnf->clausify(a);
    dpll->set_frozen(var(l));
    return l;
}


void IC3::send_frame(IC3Solver &s, Frame &f, SatLit label)
{
    SatLitList cls;
    
    for (size_t i = 0; i < f.size(); ++i) {
        if (f[i].is_active()) {
            cls.clear();
            const Cube &c = f[i].get_cube();
            if (label != satLit_Undef) {
                cls.push_back(~label);
            }
            for (size_t j = 0; j < c.size(); ++j) {
                SatLit l = s.cnf->clausify(c[j]);
                cls.push_back(~l);
            }
            s.solver->add_clause(cls);
        }
    }
}


void IC3::send_statevars(IC3Solver &s)
{
    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it) {
        Aig v = *it;
        SatLit l = s.cnf->clausify(v);
        s.solver->set_frozen(var(l));
    }
}


void IC3::send_inputvars(IC3Solver &s)
{
    for (AigList::const_iterator it = model_->inputs().begin(),
             end = model_->inputs().end(); it != end; ++it) {
        Aig a = *it;
        SatLit l = s.cnf->clausify(a);
        s.solver->set_frozen(var(l));
    }
}


void IC3::reset_trans_solver(size_t i)
{
    IC3Solver &ts = get_trans_solver(i);
    if (ts.counter >= solver_reset_frequency_) {
        TimeKeeper tk(reset_time_);
        ++num_reset_;

        if (verb_ >= 2) {
            std::cout << "resetting trans solver " << i << std::endl;
        }

        size_t minframe = ts.minframe;
        ts.init(model_, factory_, opts_.cnf_simple);
        ts.minframe = minframe;
        send_statevars(ts);
        send_inputvars(ts);
        for (size_t j = 0; j < frames_.size(); ++j) {
            ts.addlabel();
        }

        if (ts.minframe == 0) {
            send_init(ts, ts.labels[0]);
        }
        for (size_t j = ts.minframe; j < frames_.size(); ++j) {
            send_frame(ts, frames_[j], ts.labels[j]);
        }

        if (opts_.sat_preprocess) {
            send_trans(ts, satLit_Undef);
            {
                TimeKeeper tk(preprocess_time_);
                ts.solver->preprocess();
            }
        }
    }
}


inline bool IC3::solve(IC3Solver &s)
{
    bool sat = false;
    double curtime = 0;
    ++num_solve_calls_;
    {
        TimeKeeper t(curtime);
        sat = s.solver->solve(assumptions_);
    }
    solve_time_ += curtime;
    if (sat) {
        ++num_solve_sat_calls_;
        solve_sat_time_ += curtime;
    } else {
        ++num_solve_unsat_calls_;
        solve_unsat_time_ += curtime;
    }
    num_solve_conflicts_ += s.solver->get_last_conflicts();
    num_solve_decisions_ += s.solver->get_last_decisions();
    num_solve_propagations_ += s.solver->get_last_propagations();
    
    return sat;
}


void IC3::new_frame()
{
    frames_.push_back(Frame());
    frame_inactives_.push_back(0);

    for (size_t i = 1; i < solvers_.size()-1; ++i) {
        solvers_[i].addlabel();
    }
    
    if (opts_.max_trans_solvers <= 0 ||
        solvers_.size()-2 < opts_.max_trans_solvers) {
        IC3Solver &trans_solver = solvers_.back();
        trans_solver.init(model_, factory_, opts_.cnf_simple);
        for (size_t i = 0; i < frames_.size(); ++i) {
            trans_solver.addlabel();
        }
        
        send_statevars(trans_solver);
        send_inputvars(trans_solver);

        size_t minframe = depth() > 0 ? depth() - 1 : depth();
        trans_solver.minframe = minframe;
        if (minframe == 0) {
            send_init(trans_solver, trans_solver.labels[0]);
        }
        for (size_t j = minframe; j < frames_.size(); ++j) {
            send_frame(trans_solver, frames_[j],
                       trans_solver.labels[j]);
        }

        if (opts_.sat_preprocess) {
            send_trans(trans_solver, satLit_Undef);
            {
                TimeKeeper tk(preprocess_time_);
                trans_solver.solver->preprocess();
            }
        }

        solvers_.push_back(IC3Solver());
        IC3Solver &prop_solver = solvers_.back();
        prop_solver.init(model_, factory_, opts_.cnf_simple);
        send_statevars(prop_solver);
        send_inputvars(prop_solver);
        prop_label_ = get_prop(prop_solver, un_, opts_.prop_unroll);
        prop_solver.addlabel();
    } else {
        // reached bound on number of solvers
        // 
        // if we have at most N solvers, we use 1 solver per frame for the
        // last N-1 frames, and one solver for the rest
        size_t n = opts_.max_trans_solvers;

        IC3Solver &trans_solver = solvers_[1];
        size_t idx = depth() - (n-1);
        assert(idx < frames_.size());
        
        send_frame(trans_solver, frames_[idx], trans_solver.labels[idx]);

        for (size_t i = 1; i < n; ++i) {
            IC3Solver &ts = solvers_[1+i];
            ts.init(model_, factory_, opts_.cnf_simple);
            for (size_t j = 0; j < frames_.size(); ++j) {
                ts.addlabel();
            }

            send_statevars(ts);
            send_inputvars(ts);

            size_t minframe = depth() - (n - i);
            ts.minframe = minframe;
            if (minframe == 0) {
                send_init(ts, ts.labels[0]);
            }
            for (size_t j = minframe; j < frames_.size(); ++j) {
                send_frame(ts, frames_[j], ts.labels[j]);
            }
        }

        IC3Solver &prop_solver = solvers_.back();
        prop_solver.init(model_, factory_, opts_.cnf_simple);
        send_statevars(prop_solver);
        send_inputvars(prop_solver);
        prop_label_ = get_prop(prop_solver, un_, opts_.prop_unroll);
        prop_solver.addlabel();
    }
}


bool IC3::check_init()
{
    SatSolver *s = factory_->new_satsolver();
    CnfConv cnf(model_, s);
    IC3Solver solver;
    solver.solver = s;
    solver.cnf = &cnf;
    send_init(solver, satLit_Undef);
    SatLitList badcls;
    for (int k = 0; k <= opts_.prop_unroll; ++k) {
        SatLit l = get_prop(solver, un_, k);
        badcls.push_back(~l);
    }
    s->add_clause(badcls);

    assumptions_.clear();
    bool sat = solve(solver);

    if (sat) {
        size_t depth = 0;
        for ( ; depth < badcls.size(); ++depth) {
            SatLit l = badcls[depth];
            SatValue val = s->get_model_value(l);
            if (val == sat_True) {
                break;
            }
        }

        for (size_t i = 0; i <= depth; ++i) {
            cex_trace_.push_back(Cube());
            Cube &c = cex_trace_.back();

            for (AigList::const_iterator it = model_->statevars().begin(),
                     end = model_->statevars().end(); it != end; ++it) {
                Aig a = *it;
                Aig ua = un_.unroll(a, i);
                SatLit l = cnf.lookup(ua);
                if (l != satLit_Undef) {
                    SatValue val = s->get_model_value(l);
                    if (val != sat_Undef) {
                        c.push_back(aig_lit(a, val == sat_False));
                    }
                }
            }

            for (AigList::const_iterator it = model_->inputs().begin(),
                     end = model_->inputs().end(); it != end; ++it) {
                Aig a = *it;
                Aig ua = un_.unroll(a, i);
                SatLit l = cnf.lookup(ua);
                if (l != satLit_Undef) {
                    SatValue val = s->get_model_value(l);
                    if (val != sat_Undef) {
                        c.push_back(aig_lit(a, val == sat_False));
                    }
                }
            }

            sort(c, cube_lt);
        }

        cex_trace_ok_ = true;
    }
    
    delete s;
    return !sat;
}


bool IC3::get_bad_cube(Cube *out)
{
    assumptions_.clear();
    IC3Solver &s = get_prop_solver(true);
    bool sat = solve(s);

    if (sat) {
        Cube cur;
        Cube inputs;
        extract_model(depth(), &cur, &inputs);
        generalize_model(depth(), cur, inputs, NULL, out);
        assert(!is_initial(*out));
        return true;
    } else {
        return false;
    }
}


void IC3::extract_model(int time, Cube *out_state, Cube *out_inputs)
{
    IC3Solver &s = (time == depth()) ? get_prop_solver() :
        get_trans_solver(time);
    
    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it) {
        Aig a = *it;
        SatLit l = aig2lit(s, a);
        SatValue val = s.solver->get_model_value(l);
        if (val != sat_Undef) {
            out_state->push_back(aig_lit(a, val == sat_False));
        }
    }
    if (time == depth()) {
        for (int k = 0; k <= opts_.prop_unroll; ++k) {
            for (AigList::const_iterator it = model_->inputs().begin(),
                     end = model_->inputs().end(); it != end; ++it) {
                Aig a = *it;
                a = un_.unroll(a, k);
                SatLit l = s.cnf->lookup(a);
                if (l != satLit_Undef) {
                    SatValue val = s.solver->get_model_value(l);
                    if (val != sat_Undef) {
                        out_inputs->push_back(aig_lit(a, val == sat_False));
                    }
                }
            }
        }
    } else {
        for (AigList::const_iterator it = model_->inputs().begin(),
                 end = model_->inputs().end(); it != end; ++it) {
            Aig a = *it;
            SatLit l = s.cnf->lookup(a);
            SatValue val = s.solver->get_model_value(l);
            if (val != sat_Undef) {
                out_inputs->push_back(aig_lit(a, val == sat_False));
            }
        }
    }
}


void IC3::generalize_model(int time, const Cube &cur, const Cube &inputs,
                           const Cube *badstate, Cube *out)
{
    TimeKeeper t(generalize_model_time_);

    bool skip = !use_generalize_;
    if (skip) {
        *out = cur;
        return;
    }

    assumptions_.clear();
    IC3Solver &s = badstate ? get_trans_solver(time, false) :
        get_prop_solver(false);
   
    if (badstate) {
        s.solver->push();
        add_cube_as_clause(s, *badstate);
        ++s.counter;
    }
    Cube fix = inputs;

    size_t max_failures = 2;
    size_t num_failures = 0;
    size_t max_iters = 32;
    
    SatLitSet core;

    out->assign(cur.begin(), cur.end());

    size_t limit = assumptions_.size();
    
    for (size_t i = 0; num_failures < max_failures && i < max_iters; ++i) {
        assumptions_.resize(limit);
        size_t size_before = out->size();
        if (i > 0) {
            random_shuffle(rng_, *out);
            random_shuffle(rng_, fix);
        }
        for (size_t j = 0; j < fix.size(); ++j) {
            SatLit l = aig2lit(s, fix[j]);
            assumptions_.push_back(l);
        }
        size_t outstart = assumptions_.size();
        for (size_t j = 0; j < out->size(); ++j) {
            SatLit l = aig2lit(s, (*out)[j]);
            assumptions_.push_back(l);
        }

        bool sat = solve(s);
        if (sat) {
            sort(*out, cube_lt);
            break;
        }
        
        assert(!sat);

        core.clear();
        s.solver->get_unsat_core(core);
        size_t j, k;
        for (j = k = 0; j < out->size(); ++j) {
            SatLit l = assumptions_[outstart + j];
            if (core.find(~l) != core.end()) {
                (*out)[k++] = (*out)[j];
            }
        }
        out->resize(k);
        if (out->size() >= size_before) {
            ++num_failures;
        }
    }

    if (badstate) {
        s.solver->pop();
    }
}


bool IC3::is_cex(ProofObligationQueue &q)
{
    return q.top()->get_time() == 0;
}


bool IC3::rec_block_cube(const Cube &c, int time)
{
    TimeKeeper t(rec_block_cube_time_);
    
    ProofObligationQueue q;
    std::vector<Cube *> to_delete;
    std::vector<ProofObligation *> po_to_delete;

    int pidx = 1;
    ProofObligation *po = new ProofObligation(&c, time, pidx++);
    po_to_delete.push_back(po);
    q.push(po);
    
    bool first = true;
    bool retval = true;
    Cube inputs;

    while (!q.empty()) {
        if (rec_block_iter_++ == 200) {
            rec_block_iter_ = 0;
            print_frames(0);
        }
        if (solver_reset_frequency_) {
            for (size_t i = 0; i < depth(); ++i) {
                reset_trans_solver(i);
            }
        }

        ProofObligation *s = NULL;

        if (is_cex(q)) {
            if (verb_ >= 2) {
                std::cout << "found counterexample" << std::endl;
            }
            retval = false; // counterexample found

            s = q.top();
            for (const ProofObligation *cur = s; cur; cur = cur->next()) {
                const Cube &c = cur->get_cube();
                if (cur == s) {
                    cex_trace_.push_back(c);
                } else {
                    cex_trace_.push_back(Cube());
                }
                Cube &cc = cex_trace_.back();
                const Cube &i = cur->get_inputs();
                cc.insert(cc.end(), i.begin(), i.end());
            }
            break;
        } else if (!q.empty()) {
            s = q.top();
        } else {
            break;
        }
        
        if (first || !is_blocked(s->get_cube(), s->get_time())) {
            first = false;

            Cube *c = new Cube();
            int time;
            if (try_blocking(s->get_cube(), s->get_time(), c, &time, true,
                             &inputs)) {
                q.pop();
                // cube 's' was blocked by image of predecessor
                // generalize it and push it forward
                generalize_and_push_forward(*c, time);
                add_blocked_cube(*c, time);
                delete c;
                if (!opts_.short_cex &&
                    s->get_time() < depth()) {
                    po = new ProofObligation(&(s->get_cube()), s->get_time()+1,
                                             pidx++);
                    po->set_next(s->next());
                    po->set_inputs(s->get_inputs());
                    
                    q.push(po);
                    po_to_delete.push_back(po);
                }
            } else {
                // cube 's' was not blocked by image of predecessor
                if (verb_ >= 2) {
                    std::cout << "enqueuing proof obligation for cube of size "
                              << c->size() << " at depth " << (s->get_time()-1)
                              << std::endl;
                }
                to_delete.push_back(c);
                po = new ProofObligation(c, s->get_time()-1, pidx++);
                po->set_next(s);
                std::swap(po->inputs(), inputs);

                q.push(po);
                po_to_delete.push_back(po);
            }
        } else {
            if (verb_ >= 2) {
                std::cout << "cube of size " << s->get_cube().size()
                          << " at depth " << s->get_time()
                          << " is already blocked, dequeuing" << std::endl;
            }
            q.pop();
        }
    }

    for (size_t i = po_to_delete.size(); i > 0; --i) {
        ProofObligation *po = po_to_delete[i-1];
        delete po;
    }

    for (size_t i = to_delete.size(); i > 0; --i) {
        Cube *c = to_delete[i-1];
        delete c;
    }

    return retval;
}


bool IC3::is_blocked(const Cube &c, int time)
{
    FrameCube pc(c);
    
    for (size_t i = time; i < frames_.size(); ++i) {
        Frame &f = frames_[i];
        for (size_t j = 0; j < f.size(); ++j) {
            FrameCube &fj = f[j];
            if (fj.is_active() && fj.subsumes(pc)) {
                return true;
            }
        }
    }

    return false;
}


bool IC3::is_initial(const Cube &c)
{
    IC3Solver &s = get_init_solver();
    assumptions_.clear();
    for (size_t i = 0; i < c.size(); ++i) {
        assumptions_.push_back(aig2lit(s, c[i]));
    }
    bool sat = solve(s);

    return sat;
}


bool IC3::try_blocking(const Cube &c, int time, Cube *out, int *out_time,
                       bool compute_model, Cube *out_inputs)
{
    TimeKeeper t(try_blocking_time_);
    ++num_try_blocking_;
    
    assert(time > 0);

    assumptions_.clear();
    IC3Solver &s = get_trans_solver(time-1, true);

    Cube primed;
    get_next(c, &primed, time-1);
    {
        TimeKeeper tk(clausify_time_);
        
        for (size_t i = 0; i < primed.size(); ++i) {
            SatLit l = s.cnf->clausify(primed[i]);
            assumptions_.push_back(l);
        }
    }
    SatLit lbl = satLit_Undef;
    bool sat = true;

    if (opts_.use_relind) {
        s.solver->push();
        add_cube_as_clause(s, c);
        ++s.counter;
    }
    sat = solve(s);
    
    if (!sat) {
        if (!out) {
            if (opts_.use_relind) {
                s.solver->pop();
            }
            return true;
        }
        
        SatLitSet core;
        s.solver->get_unsat_core(core);
        if (opts_.use_relind) {
            s.solver->pop();
        }

        // try minimizing using the unsat core
        Cube candidate;
        for (size_t i = 0; i < primed.size(); ++i) {
            Aig a = primed[i];
            if (core.find(~(s.cnf->lookup(a))) != core.end()) {
                candidate.push_back(c[i]);
            }
        }
        IC3Solver &i_s = get_init_solver();
        assumptions_.clear();
        for (size_t i = 0; i < candidate.size(); ++i) {
            assumptions_.push_back(aig2lit(i_s, candidate[i]));
        }

        sat = solve(i_s);
        if (sat) {
            // minimize by removing literals one at a time
            for (size_t i = 0; i < candidate.size(); ++i) {
                assumptions_.pop_back();
            }
            candidate.clear();
            for (size_t i = 0; i < c.size(); ++i) {
                candidate.push_back(c[i]);
            }
            while (true) {
                int removed = -1;
                for (size_t i = 0; i < candidate.size(); ++i) {
                    Aig o = candidate[i];
                    if (o == AigManager::aig_null()) {
                        continue;
                    }
                    Aig l = primed[i];
                    if (core.find(~aig2lit(s, l)) == core.end()) {
                        // try removing o from out
                        size_t n = 0;
                        for (size_t j = 0; j < candidate.size(); ++j) {
                            if (j != i &&
                                candidate[j] != AigManager::aig_null()) {
                                assumptions_.push_back(
                                    aig2lit(i_s, candidate[j]));
                                ++n;
                            }
                        }
                        sat = solve(i_s);
                        assumptions_.resize(assumptions_.size()-n);

                        if (!sat) {
                            removed = i;
                            break;
                        }
                    }
                }
                if (removed >= 0) {
                    candidate[removed] = AigManager::aig_null();
                } else {
                    // ok, we have removed as many as possible
                    // filter out the satLit_Undef entries from *out
                    size_t i, j;
                    for (i = j = 0; i < candidate.size(); ++i) {
                        if (candidate[i] != AigManager::aig_null()) {
                            out->push_back(candidate[i]);
                        }
                    }
                    break;
                }
            }
            *out_time = time;
        } else {
            *out = candidate;
            *out_time = -1;
            for (size_t i = time-1; i < frames_.size()-1; ++i) {
                if (core.find(~s.labels[i]) != core.end()) {
                    *out_time = i+1;
                    break;
                }
            }
            if (*out_time < 0) {
                *out_time = depth();
            }
            assert(*out_time >= time);
        }
        
        return true;
    } else {
        
        if (!compute_model) {
            if (opts_.use_relind) {
                s.solver->pop();
            }
            return false;
        }
        
        // build a counterexample to induction
        Cube prev;
        Cube inputs;
        extract_model(time-1, &prev, &inputs);

        if (opts_.use_relind) {
            s.solver->pop();
        }

        out->clear();
        generalize_model(time-1, prev, inputs, &primed, out);

        assert(!is_blocked(*out, time-1));

        if (out_inputs) {
            std::swap(*out_inputs, inputs);
        }
        
        return false;
    }
}


void IC3::generalize_and_push_forward(Cube &c, int &time)
{
    TimeKeeper t(generalize_and_push_time_);

    if (opts_.use_ctg) {
        generalize_ctg(c, time, 1);
    } else {
        generalize(c, time);
    }
    push_forward(c, time);
}


void IC3::generalize(Cube &c, int &time)
{
    Cube tmp;
    int timetmp;

    Cube cand = c;
    sort(cand, ActLt(activity_));

    size_t reset = 0;
    for (size_t i = 0; c.size() > 1 && i < cand.size(); ++i) {
        size_t index = reset + i;
        if (index >= cand.size()) {
            index -= cand.size();
        }
        Aig elem = cand[index];
        tmp.clear();
        bool found = false;
        for (size_t j = 0; j < c.size(); ++j) {
            if (elem == c[j]) {
                found = true;
            } else {
                tmp.push_back(c[j]);
            }
        }
        if (found && !is_initial(tmp)) {
            if (try_blocking(tmp, time, &tmp, &timetmp, false)) {
                std::swap(tmp, c);
                time = timetmp;
                reset = index;
                i = 0;
            }
        }
    }
}


void IC3::push_forward(Cube &c, int &time)
{
    Cube tmp;
    int timetmp;

    while (time < depth()-1) {
        tmp.clear();
        if (try_blocking(c, time+1, &tmp, &timetmp, false)) {
            if (verb_ >= 2) {
                std::cout << "pushed forward from size " << c.size()
                          << " at depth " << time
                          << " to size " << tmp.size()
                          << " at depth " << timetmp << std::endl;
            }
            std::swap(tmp, c);
            time = timetmp;
        } else {
            break;
        }
    }
}


// CTG-based generalization as described in the paper
// 
//   Better Generalization in IC3, by Hassan, Bradley and Somenzi, FMCAD'13
//
void IC3::generalize_ctg(Cube &c, int &time, int rec_level)
{
    Cube tmp;
    
    for (size_t i = 0; i < c.size(); ) {
        tmp.clear();
        for (size_t j = 0; j < c.size(); ++j) {
            if (i != j) {
                tmp.push_back(c[j]);
            }
        }

        if (ctg_down(tmp, time, rec_level)) {
            std::swap(tmp, c);
        } else {
            ++i;
        }
    }
}


bool IC3::ctg_down(Cube &c, int &time, int rec_level)
{
    const size_t max_ctgs = 3;
    const int max_rec_level = 1;
    
    size_t ctgs = 0;
    size_t joins = 0;

    Cube tmp;
    int timetmp;

    while (true) {
        if (is_initial(c)) {
            return false;
        }
        if (try_blocking(c, time, &tmp, &timetmp, true)) {
            std::swap(c, tmp);
            time = timetmp;
            return true;
        } else if (rec_level > max_rec_level) {
            return false;
        } else if (ctgs < max_ctgs && time > 0) {
            TimeKeeper t(ctg_time_);
            if (!is_initial(tmp) &&
                try_blocking(tmp, time-1, &tmp, &timetmp, false)) {
                ++num_added_ctgs_;
                ++ctgs;
                push_forward(tmp, timetmp);
                generalize_ctg(tmp, timetmp, rec_level+1);
                add_blocked_cube(tmp, timetmp);
                continue;
            }
        }
        ctgs = 0;
        ++joins;
        
        size_t j = 0;
        for (size_t i = 0, k = 0; i < c.size() && k < tmp.size(); ) {
            if (c[i] == tmp[k]) {
                c[j++] = c[i];
                ++i;
                ++k;
            } else if (cube_lt(c[i], tmp[k])) {
                ++i;
            } else {
                ++k;
            }
        }
        c.resize(j);
    }
}


void IC3::add_blocked_cube(Cube &c, int time)
{
    FrameCube pc(c);

    // remove subsumed clauses
    for (size_t d = 1; d < time+1; ++d) {
        Frame &fd = frames_[d];
        for (size_t i = 0; i < fd.size(); ++i) {
            if (fd[i].is_active() && pc.subsumes(fd[i])) {
                ++num_subsumed_cubes_;
                fd[i].set_active(false);
                ++frame_inactives_[d];
                for (size_t k = d; k < frames_.size(); ++k) {
                    ++get_trans_solver(k).counter;
                }

                for (size_t j = 0; j < fd[i].get_cube().size(); ++j) {
                    Aig v = aig_var(fd[i].get_cube()[j]);
                    activity_[v] -= 1.0;
                }
            }
        }
        if (use_gc_frame_ && frame_inactives_[d] * 2 >= fd.size()) {
            gc_frame(d);
        }
    }

    add_frame_clause(time, pc);
    ++num_added_cubes_;
}


void IC3::add_frame_clause(int time, FrameCube &pc)
{    
    frames_[time].push_back(pc);
    size_t previdx = solvers_.size();
    for (size_t i = 1; i < solvers_.size()-1; ++i) {
        IC3Solver &s = solvers_[i];
        if (s.minframe <= time) {
            add_cube_as_clause(s, pc.get_cube(), ~s.labels[time]);
        }
    }
    if (time == depth()) {
        IC3Solver &s = get_prop_solver();
        add_cube_as_clause(s, pc.get_cube(), ~s.labels[0]);
    }

    for (size_t i = 0; i < pc.get_cube().size(); ++i) {
        Aig v = aig_var(pc.get_cube()[i]);
        activity_[v] += 1.0;
    }
}


void IC3::add_cube_as_clause(IC3Solver &s, const Cube &c, SatLit lbl)
{
    SatLitList tmp;
    if (lbl != satLit_Undef) {
        tmp.push_back(lbl);
    }
    for (size_t i = 0; i < c.size(); ++i) {
        SatLit l = aig2lit(s, c[i]);
        tmp.push_back(~l);
    }
    s.solver->add_clause(tmp);
}


inline void IC3::add_cube_as_clause(IC3Solver &s, const Cube &c)
{
    add_cube_as_clause(s, c, satLit_Undef);
}


void IC3::get_next(const Cube &c, Cube *out, int time)
{
    for (size_t i = 0; i < c.size(); ++i) {
        Aig n = model_->next_value(aig_var(c[i]));
        Aig nl = aig_lit(n, aig_sign(c[i]));
        out->push_back(nl);
    }
}


bool IC3::propagate_blocked_cubes()
{
    TimeKeeper t(propagate_forward_time_);
    
    std::vector<Cube> to_add_cubes;
    std::vector<int> to_add_times;

    if (verb_ >= 2) {
        std::cout << "forward propagation" << std::endl;
    }

    for (size_t k = 1; k < depth(); ++k) {
        to_add_cubes.clear();
        to_add_times.clear();
        Frame &f = frames_[k];
        for (size_t i = 0; i < f.size(); ++i) {
            if (f[i].is_active()) {
                to_add_cubes.push_back(Cube());
                to_add_times.push_back(0);
                if (!try_blocking(f[i].get_cube(), k+1, &to_add_cubes.back(),
                                  &to_add_times.back(), false)) {
                    to_add_cubes.pop_back();
                    to_add_times.pop_back();
                }
            }
        }
        if (to_add_cubes.size() > 0) {
            if (verb_ >= 2) {
                std::cout << "propagating " << to_add_cubes.size() << "/"
                          << frames_[k].size() << " cubes from " << k
                          << " to " << (k+1) << std::endl;
            }
        }
        for (size_t i = 0; i < to_add_cubes.size(); ++i) {
            add_blocked_cube(to_add_cubes[i], to_add_times[i]);
        }

        if (fixpoint(k)) {
            return true;
        }
    }

    return false;
}


inline bool IC3::fixpoint(int time)
{
    return frames_[time].size() <= frame_inactives_[time];
}


void IC3::gc_frame(int time)
{
    ++num_gc_frame_;

    SatLitList tmp;
    Frame &f = frames_[time];
    size_t i, j;
    for (i = j = 0; i < f.size(); ++i) {
        FrameCube &c = f[i];
        if (c.is_active()) {
            f[j++] = c;
        }
    }
    f.resize(j);
    frame_inactives_[time] = 0;
}


bool IC3::get_counterexample_trace(std::vector<AigList> *out)
{
    if (cex_trace_.empty()) {
        return false;
    }

    if (cex_trace_ok_) {
        *out = cex_trace_;
        return true;
    }

    ModelSim sim(model_);
    std::vector<ModelSim::value> vals;

    AigList *cur = &(cex_trace_[0]);
    AigList state;
    ModelSim::ValMap vmap;

    for (size_t i = 0; i < cur->size(); ++i) {
        Aig a = (*cur)[i];
        Aig v = aig_var(a);
        if (model_->is_statevar(v)) {
            vmap[v] = aig_sign(a) ? ModelSim::FALSE : ModelSim::TRUE;
        }
    }
    for (AigList::const_iterator it = model_->statevars().begin(),
             end = model_->statevars().end(); it != end; ++it) {
        Aig v = *it;
        ModelSim::ValMap::iterator i = vmap.find(v);
        if (i != vmap.end()) {
            vals.push_back(i->second);
        } else {
            Aig f = model_->init_formula(v);
            if (aig_var(f) == v) {
                vals.push_back(aig_sign(f) ? ModelSim::FALSE : ModelSim::TRUE);
            } else {
                vals.push_back(ModelSim::UNDEF);
            }
        }
    }
    sim.init(vals);

    size_t idx = 0;
    while (true) {
        state.clear();

        for (AigList::const_iterator it = model_->statevars().begin(),
                 end = model_->statevars().end(); it != end; ++it) {
            Aig v = *it;
            ModelSim::value val = sim.get(v);
            if (val != ModelSim::UNDEF) {
                state.push_back(aig_lit(v, val == ModelSim::FALSE));
            }
        }
        for (size_t i = 0; i < cur->size(); ++i) {
            Aig a = (*cur)[i];
            Aig v = aig_var(a);

            assert(idx == 0 || model_->is_inputvar(v));
            
            if (idx > 0 || model_->is_inputvar(v)) {
                state.push_back(a);
            } else {
                assert(std::find(state.begin(), state.end(),
                                 aig_lit(v, !aig_sign(a))) == state.end());
            }
        }

        sort(state, cube_lt);
        out->push_back(state);

        vmap.clear();
        for (size_t i = 0; i < cur->size(); ++i) {
            Aig a = (*cur)[i];
            Aig v = aig_var(a);
            if (idx > 0 || model_->is_inputvar(v)) {
                vmap[v] = aig_sign(a) ? ModelSim::FALSE : ModelSim::TRUE;
            }
        }
        vals.clear();
        for (AigList::const_iterator it = model_->inputs().begin(),
                 end = model_->inputs().end(); it != end; ++it) {
            Aig v = *it;
            ModelSim::ValMap::iterator i = vmap.find(v);
            if (i != vmap.end()) {
                vals.push_back(i->second);
            } else {
                vals.push_back(ModelSim::UNDEF);
            }
        }

        sim.step(vals);

        ++idx;

        if (idx >= cex_trace_.size()) {
            break;
        }

        cur = &(cex_trace_[idx]);
    }

    SatSolver *s = factory_->new_satsolver();
    CnfConv cl(model_, s);
    ModelUnroll un(model_);

    const AigList &c = out->back();
    SatLitList assumps;

    for (size_t j = 0; j < c.size(); ++j) {
        Aig a = c[j];
        Aig f = un.unroll(a, 0);
        SatLit l = cl.clausify(f);
        s->add_clause(l);
    }
    
    assumps.push_back(satLit_Undef);
    bool sat = false;
    int k = 0;
    for ( ; k <= opts_.prop_unroll; ++k) {
        IC3Solver solver;
        solver.solver = s;
        solver.cnf = &cl;
        SatLit l = get_prop(solver, un, k);
        assumps[0] = ~l;
        sat = s->solve(assumps);
        if (sat) {
            break;
        }
    }
    assert(sat);

    out->pop_back();
    for (int cur_k = 0; cur_k <= k; ++cur_k) {
        out->push_back(AigList());
        AigList &al = out->back();
        
        for (AigList::const_iterator it = model_->statevars().begin(),
                 end = model_->statevars().end(); it != end; ++it) {
            Aig v = *it;
            Aig f = un.unroll(v, cur_k);
            SatLit l = cl.lookup(f);
            if (l != satLit_Undef) {
                SatValue val = s->get_model_value(l);
                if (val == sat_True) {
                    al.push_back(v);
                } else if (val == sat_False) {
                    al.push_back(model_->aig_manager()->aig_not(v));
                }
            }
        }

        for (AigList::const_iterator it = model_->inputs().begin(),
                 end = model_->inputs().end(); it != end; ++it) {
            Aig v = *it;
            Aig f = un.unroll(v, cur_k);
            SatLit l = cl.lookup(f);
            if (l != satLit_Undef) {
                SatValue val = s->get_model_value(l);
                if (val == sat_True) {
                    al.push_back(v);
                } else if (val == sat_False) {
                    al.push_back(model_->aig_manager()->aig_not(v));
                }
            }
        }

        sort(al, cube_lt);
    }

    delete s;

    return true;
}


bool IC3::get_final_invariant(std::vector<AigList> *out)
{
    if (!cex_trace_.empty()) {
        return false;
    }

    bool good = false;
    for (size_t i = 1; i < frames_.size(); ++i) {
        Frame &f = frames_[i];
        if (!good) {
            if (f.size() == 0) {
                good = true;
            }
        } else {
            for (size_t j = 0; j < f.size(); ++j) {
                if (f[j].is_active()) {
                    const Cube &c = f[j].get_cube();
                    out->push_back(c);
                    Cube &cc = out->back();

                    for (size_t n = 0; n < cc.size(); ++n) {
                        cc[n] = model_->aig_manager()->aig_not(cc[n]);
                    }
                }
            }
        }
    }

    if (opts_.prop_unroll > 0) {
        // we have to check that the clauses imply the property
        // combinationally
        SatSolver *s = factory_->new_satsolver();
        CnfConv cl(model_, s);
        SatLitList cls;

        for (size_t i = 0; i < out->size(); ++i) {
            AigList &c = (*out)[i];
            cls.clear();
            for (size_t j = 0; j < c.size(); ++j) {
                SatLit l = cl.clausify(c[j]);
                cls.push_back(l);
            }
            s->add_clause(cls);
        }

        Aig p = prop_aig();
        SatLit good = cl.clausify(p);
        s->add_clause(~good);
        bool sat = s->solve(SatLitList());
        delete s;

        if (sat) { // the invariant is not strong enough -- complete it with
                   // another IC3
            if (verb_) {
                std::cout << "completing final invariant with sub-IC3 call..."
                          << std::endl;
            }
            Options opts = opts_;
            opts.verbosity = 0;
            opts.prop_unroll = 0;
            IC3 witprover(opts, model_, prop_idx_);
            witprover.set_initial_clauses(*out);
            Result res = witprover.prove();
            assert(res == VERIFIED);
            out->clear();
            bool ok = witprover.get_final_invariant(out);
            assert(ok);
            return ok;
        }
    }

    return true;
}


Stats IC3::get_stats()
{
    Stats ret;
    std::ostringstream tmp;
    
#define ADDSTAT(name, val) do { tmp.str(""); \
        tmp << std::setprecision(3) << std::fixed << val;       \
        ret.push_back(std::make_pair(name, tmp.str()));         \
    } while (0)
    
    ADDSTAT("num_solve_calls", num_solve_calls_);
    ADDSTAT("num_solve_sat_calls", num_solve_sat_calls_);
    ADDSTAT("num_solve_unsat_calls", num_solve_unsat_calls_);
    if (num_solve_calls_) {
        ADDSTAT("num_solve_conflicts", num_solve_conflicts_);
        ADDSTAT("num_solve_decisions", num_solve_decisions_);
        ADDSTAT("num_solve_propagations", num_solve_propagations_);
        ADDSTAT("avg_solve_conflicts",
                uint64_t(round(double(num_solve_conflicts_) /
                               double(num_solve_calls_))));
        ADDSTAT("avg_solve_decisions",
                uint64_t(round(double(num_solve_decisions_) /
                               double(num_solve_calls_))));
        ADDSTAT("avg_solve_propagations",
                uint64_t(round(double(num_solve_propagations_) /
                               double(num_solve_calls_))));
    }
    ADDSTAT("num_added_cubes", num_added_cubes_);
    ADDSTAT("num_subsumed_cubes", num_subsumed_cubes_);
    if (opts_.use_ctg) {
        ADDSTAT("num_added_ctgs", num_added_ctgs_);
    }
    ADDSTAT("num_gc_frame", num_gc_frame_);
    ADDSTAT("num_reset", num_reset_);
    ADDSTAT("num_try_blocking", num_try_blocking_);
    ADDSTAT("rec_block_cube_time", rec_block_cube_time_);
    ADDSTAT("propagate_forward_time", propagate_forward_time_);
    ADDSTAT("generalize_model_time", generalize_model_time_);
    ADDSTAT("generalize_and_push_time", generalize_and_push_time_);
    if (opts_.use_ctg) {
        ADDSTAT("ctg_time", ctg_time_);
    }
    ADDSTAT("try_blocking_time", try_blocking_time_);
    ADDSTAT("reset_time", reset_time_);
    ADDSTAT("clausify_time", clausify_time_);
    if (opts_.sat_preprocess) {
        ADDSTAT("preprocess_time", preprocess_time_);
    }
    ADDSTAT("solve_time", solve_time_);
    ADDSTAT("solve_sat_time", solve_sat_time_);
    ADDSTAT("solve_unsat_time", solve_unsat_time_);
    ADDSTAT("prove_time", total_time_);
    size_t mem_used = get_mem_used_bytes() / (1024 * 1024);
    ADDSTAT("memory_used_mb", mem_used);

#undef ADDSTAT
    
    return ret;

}


void IC3::print_frames(int numcall)
{
    if (verb_ >= 1) {
        if (numcall) {
            std::cout << std::setw(4) << numcall << " :";
        } else {
            std::cout << " ... :";
        }
        for (size_t j = 0; j < frames_.size(); ++j) {
            size_t n = frames_[j].size() - frame_inactives_[j];
            std::cout << " " << n;
        }
        std::cout << std::endl;
    }
}


// debugging methods

std::string IC3::cube2str(const Cube &c)
{
    std::ostringstream buf;
    buf << "[ ";
    for (size_t i = 0; i < c.size(); ++i) {
        Aig a = c[i];
        if (AigManager::aig_is_negated(a)) {
            buf << "~";
        }
        buf << AigManager::aig_id(AigManager::aig_strip(a)) << " ";
    }
    buf << "]";
    return buf.str();
}


} // namespace simplic3
