// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Helper structures for the IC3 engine
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#ifndef SIMPLIC3_IC3HELPERS_H
#define SIMPLIC3_IC3HELPERS_H

#include "simplic3/cnf.h"
#include "simplic3/satsolver.h"
#include <algorithm>

namespace simplic3 {

typedef AigList Cube;

class FrameCube {
public:
    FrameCube();
    FrameCube(const Cube &cube);
    Aig operator[](size_t idx) const;
    size_t size() const;
    uint32_t abstraction() const;
    bool subsumes(const FrameCube &other) const;
    const Cube &get_cube() const;
    bool is_active() const;
    void set_active(bool a);

private:
    void calc_abstraction();

    Cube cube_;
    uint32_t abstraction_;
    bool active_;
};


class ProofObligation {
public:
    ProofObligation(const Cube *c, int time, int idx);

    const Cube &get_cube() const;
    int get_time() const;

    void set_next(ProofObligation *n);
    ProofObligation *next();
    const ProofObligation *next() const;
    size_t length() const;

    void set_inputs(const Cube &i);
    const Cube &get_inputs() const;
    Cube &inputs();

    bool operator<(const ProofObligation &other) const;

private:
    const Cube *cube_;
    Cube inputs_;
    int time_;
    int idx_;
    ProofObligation *next_;
};


class ProofObligationQueue {
public:
    ProofObligationQueue() {}
    
    void push(ProofObligation *p)
    {
        data_.push_back(p);
        std::push_heap(data_.begin(), data_.end(), lt_);
    }
    
    void pop()
    {
        std::pop_heap(data_.begin(), data_.end(), lt_);
        data_.pop_back();
    }
    
    ProofObligation *&top() { return data_[0]; }
    bool empty() const { return data_.empty(); }
    size_t size() const { return data_.size(); }
    ProofObligation *&operator[](size_t idx) { return data_[idx]; }
    void clear() { data_.clear(); }

private:
    std::vector<ProofObligation *> data_;
    struct ProofObligation_lt {
        bool operator()(const ProofObligation *a,
                        const ProofObligation *b) const
        { return (*a) < (*b); }
    };
    ProofObligation_lt lt_;
};


typedef std::vector<FrameCube> Frame;
typedef std::vector<Frame> FrameList;


struct IC3Solver {
    SatSolver *solver;
    CnfConv *cnf;
    SatLitList labels;
    size_t counter;
    size_t minframe;

    IC3Solver(): solver(NULL), cnf(NULL), counter(0), minframe(0) {}

    void init(Model *model, SatSolverFactory *factory, bool cnf_simple)
    {
        destroy();
        solver = factory->new_satsolver();
        cnf = new CnfConv(model, solver, cnf_simple);
        counter = 0;
        minframe = 0;
    }
    
    void destroy()
    {
        if (cnf) {
            delete cnf;
            cnf = NULL;
        }
        if (solver) {
            delete solver;
            solver = NULL;
        }
        labels.clear();
    }

    void addlabel()
    {
        SatVar v = solver->new_var();
        solver->set_frozen(v);
        labels.push_back(SatLit(v));
    }
};


//-----------------------------------------------------------------------------
// Implementation of inline methods
//-----------------------------------------------------------------------------

inline const Cube &ProofObligation::get_cube() const { return *cube_; }
inline int ProofObligation::get_time() const { return time_; }

inline void ProofObligation::set_next(ProofObligation *n) { next_ = n; }
inline ProofObligation *ProofObligation::next() { return next_; }
inline const ProofObligation *ProofObligation::next() const { return next_; }

inline bool ProofObligation::operator<(const ProofObligation &other) const
{
    if (time_ != other.time_) {
        return time_ > other.time_;
    } else {
        return idx_ < other.idx_;
    }
}


inline void ProofObligation::set_inputs(const Cube &i) { inputs_ = i; }
inline const Cube &ProofObligation::get_inputs() const { return inputs_; }
inline Cube &ProofObligation::inputs() { return inputs_; }

inline Aig FrameCube::operator[](size_t idx) const { return cube_[idx]; }
inline size_t FrameCube::size() const { return cube_.size(); }
inline uint32_t FrameCube::abstraction() const { return abstraction_; }
inline const Cube &FrameCube::get_cube() const { return cube_; }
inline bool FrameCube::is_active() const { return active_; }
inline void FrameCube::set_active(bool a) { active_ = a; }



} // namespace simplic3

#endif // SIMPLIC3_IC3HELPERS_H
