// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// MiniSat SAT solver backend
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "minisatsolver.h"
#include <assert.h>

// work around macro definitions of l_True etc. in Minisat-2.2.0
#ifdef l_True
  #define Minisat_l_True l_True
  #define Minisat_l_False l_False
  #define Minisat_l_Undef l_Undef
  using Minisat::lbool;
#else // ifdef l_True
  #define Minisat_l_True Minisat::l_True
  #define Minisat_l_False Minisat::l_False
  #define Minisat_l_Undef Minisat::l_Undef
#endif // ifdef l_True

namespace simplic3 {


//-----------------------------------------------------------------------------
// MinisatSolver
//-----------------------------------------------------------------------------

MinisatSolver::MinisatSolver(bool simp):
    simp_(simp)
{
    dpll_ = new Minisat::SimpSolver();
    if (!simp_) {
        dpll_->eliminate(true);
    }
    in_push_ = false;
    pushlbl_ = Minisat::lit_Undef;
}


MinisatSolver::~MinisatSolver()
{
    delete dpll_;
}


namespace {

inline SatVar var2satvar(Minisat::Var v) { return v+1; }
inline Minisat::Var satvar2var(SatVar v) { return v-1; }

inline SatLit lit2satlit(Minisat::Lit l)
{ return SatLit(var2satvar(Minisat::var(l)), Minisat::sign(l)); }

inline Minisat::Lit satlit2lit(SatLit l)
{ return Minisat::mkLit(satvar2var(var(l)), sign(l)); }

inline SatValue lbool2val(Minisat::lbool l)
{
    if (l == Minisat_l_True) {
        return sat_True;
    } else if (l == Minisat_l_False) {
        return sat_False;
    } else {
        return sat_Undef;
    }
}

inline Minisat::lbool val2lbool(SatValue l)
{
    if (l == sat_True) {
        return Minisat_l_True;
    } else if (l == sat_False) {
        return Minisat_l_False;
    } else {
        return Minisat_l_Undef;
    }
}

} // namespace


SatVar MinisatSolver::new_var()
{
    Minisat::Var v = dpll_->newVar();
    return var2satvar(v);
}


void MinisatSolver::add_clause(const SatLitList &cls, SatLit label)
{
    tmplist_.clear();
    tmplist_.capacity(cls.size() + int(label != satLit_Undef));
    for (size_t i = 0; i < cls.size(); ++i) {
        tmplist_.push(satlit2lit(cls[i]));
    }
    if (label != satLit_Undef) {
        tmplist_.push(satlit2lit(label));
    }
    if (in_push_) {
        tmplist_.push(~pushlbl_);
    }
    dpll_->addClause_(tmplist_);
}


void MinisatSolver::add_clause(const SatLitList &cls)
{
    add_clause(cls, satLit_Undef);
}


bool MinisatSolver::solve(const SatLitList &assumps)
{
    tmplist_.clear();
    tmplist_.capacity(assumps.size() + size_t(in_push_));
    if (in_push_) {
        tmplist_.push(pushlbl_);
    }
    for (size_t i = 0; i < assumps.size(); ++i) {
        tmplist_.push(satlit2lit(assumps[i]));
    }

    dpll_->propagations = 0;
    dpll_->conflicts = 0;
    dpll_->decisions = 0;
    
    return dpll_->solve(tmplist_, false, true);
}


SatValue MinisatSolver::get_model_value(SatLit l)
{
    Minisat::lbool val = dpll_->modelValue(satlit2lit(l));
    return lbool2val(val);
}


void MinisatSolver::get_unsat_core(SatLitSet &out)
{
    for (size_t i = 0; i < dpll_->conflict.size(); ++i) {
        if (in_push_ &&
            Minisat::var(dpll_->conflict[i]) == Minisat::var(pushlbl_)) {
            continue;
        }
        out.insert(lit2satlit(dpll_->conflict[i]));
    }
}


void MinisatSolver::push()
{
    in_push_ = true;
    pushlbl_ = Minisat::mkLit(dpll_->newVar());
    if (simp_) {
        dpll_->setFrozen(Minisat::var(pushlbl_), true);
    }
}


void MinisatSolver::pop()
{
    in_push_ = false;
    dpll_->releaseVar(~pushlbl_);
    pushlbl_ = Minisat::lit_Undef;
}


void MinisatSolver::set_frozen(SatVar v)
{
    if (simp_) {
        dpll_->setFrozen(satvar2var(v), true);
    }
}


void MinisatSolver::unfreeze_all()
{
    if (simp_) {
        for (Minisat::Var v = 0; v < dpll_->nVars(); ++v) {
            dpll_->setFrozen(v, false);
        }
    }
}


void MinisatSolver::preprocess()
{
    if (simp_) {
        dpll_->eliminate(true);
    }
}

//-----------------------------------------------------------------------------
// MinisatSolverFactory
//-----------------------------------------------------------------------------

MinisatSolverFactory::MinisatSolverFactory(bool simp):
    simp_(simp)
{
}

SatSolver *MinisatSolverFactory::new_satsolver()
{
    return new MinisatSolver(simp_);
}


} // namespace simplic3
