// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Model (finite state machine) representation
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "model.h"
#include <assert.h>
#include <fstream>
#include <algorithm>

namespace simplic3 {

Model::Model():
    mgr_()
{
    nextvar_ = 0;
}


Model::~Model()
{
}


const AigList &Model::inputs() const
{
    return inputs_;
}


const AigList &Model::properties() const
{
    return properties_;
}


const AigList &Model::statevars() const
{
    return statevars_;
}


bool Model::is_statevar(Aig sv) const
{
    return nextmap_.find(sv) != nextmap_.end();
}


bool Model::is_inputvar(Aig e) const
{
    return inputset_.find(e) != inputset_.end();
}


bool Model::is_nextstatevar(Aig e) const
{
    return curmap_.find(e) != curmap_.end();
}


Aig Model::init_formula(Aig sv)
{
    LatchMap::const_iterator it = latchmap_.find(sv);
    if (it == latchmap_.end()) {
        return AigManager::aig_null();
    } else {
        Aig ival = it->second.init;
        Aig f = mgr_.aig_iff(sv, ival);
        return f;
    }
}


Aig Model::init_value(Aig sv) const
{
    LatchMap::const_iterator it = latchmap_.find(sv);
    if (it == latchmap_.end()) {
        return AigManager::aig_null();
    } else {
        Aig ival = it->second.init;
        return ival;
    }
}


Aig Model::next_value(Aig sv) const
{
    LatchMap::const_iterator it = latchmap_.find(sv);
    if (it == latchmap_.end()) {
        return AigManager::aig_null();
    } else {
        return it->second.trans;
    }
}


Aig Model::next(Aig sv) const
{
    NextMap::const_iterator it = nextmap_.find(sv);
    if (it == nextmap_.end()) {
        return AigManager::aig_null();
    } else {
        return it->second;
    }
}


Aig Model::cur(Aig nv) const
{
    NextMap::const_iterator it = curmap_.find(nv);
    if (it == curmap_.end()) {
        return AigManager::aig_null();
    } else {
        return it->second;
    }
}


void Model::add_aiger_latch(Aig v, Aig init, Aig next)
{
    statevars_.push_back(v);
    Aig n = newvar();
    nextmap_[v] = n;
    curmap_[n] = v;
    latchmap_[v] = Latch(init, next);
}


void Model::add_aiger_input(Aig a)
{
    inputs_.push_back(a);
    inputset_.insert(a);
}


void Model::add_aiger_output(Aig a)
{
    properties_.push_back(a);
}


typedef HashMap<unsigned int, Aig> AigerMap;

bool Model::init_from_aiger(aiger *aigmgr)
{
    if (aigmgr->num_constraints != 0 ||
        aigmgr->num_justice != 0 ||
        aigmgr->num_fairness != 0) {
        return false;
    }
    
    AigerMap cache;
    nextvar_ = aigmgr->maxvar+1;

    const unsigned char *coi = aiger_coi(aigmgr);
    
    for (unsigned int i = 0; i < aigmgr->num_inputs; ++i) {
        unsigned int lit = aigmgr->inputs[i].lit;

        if (!coi[aiger_lit2var(lit)]) {
            continue;
        }

        Aig iv = mgr_.aig_var(aiger_lit2var(lit));
        add_aiger_input(iv);
        cache[lit] = iv;
    }

    for (unsigned int i = 0; i < aigmgr->num_latches; ++i) {
        unsigned int lit = aigmgr->latches[i].lit;

        if (!coi[aiger_lit2var(lit)]) {
            continue;
        }

        Aig lv = mgr_.aig_var(aiger_lit2var(lit));
        cache[lit] = lv;
    }
    
    cache[aiger_false] = mgr_.aig_false();
    cache[aiger_true] = mgr_.aig_true();
    std::vector<unsigned int> to_process;
    
    for (unsigned int i = 0; i < aigmgr->num_ands; ++i) {
        to_process.clear();
        to_process.push_back(aigmgr->ands[i].lhs);

        while (!to_process.empty()) {
            unsigned int lit = to_process.back();
            if (cache.find(lit) != cache.end()) {
                to_process.pop_back();
                continue;
            }
            if (!coi[aiger_lit2var(lit)]) {
                to_process.pop_back();
                continue;
            }

            if (aiger_sign(lit)) {
                unsigned int arg = aiger_strip(lit);
                AigerMap::iterator it = cache.find(arg);
                if (it != cache.end()) {
                    cache[lit] = mgr_.aig_not(it->second);
                    to_process.pop_back();
                } else {
                    to_process.push_back(arg);
                }
            } else {
                aiger_and *aa = aiger_is_and(aigmgr, lit);
                assert(aa);
                Aig l = mgr_.aig_null(), r = mgr_.aig_null();
                AigerMap::iterator it = cache.find(aa->rhs0);
                if (it != cache.end()) {
                    l = it->second;
                } else {
                    to_process.push_back(aa->rhs0);
                }
                it = cache.find(aa->rhs1);
                if (it != cache.end()) {
                    r = it->second;
                } else {
                    to_process.push_back(aa->rhs1);
                }
                if (l != mgr_.aig_null() && r != mgr_.aig_null()) {
                    to_process.pop_back();
                    assert(lit == aa->lhs);
                    cache[lit] = mgr_.aig_and(l, r);
                }
            }
        }
    }

    for (unsigned int i = 0; i < aigmgr->num_latches; ++i) {
        unsigned int lit = aigmgr->latches[i].lit;
        if (!coi[aiger_lit2var(lit)]) {
            continue;
        }
        unsigned int n = aigmgr->latches[i].next;
        bool neg = aiger_sign(n);
        if (neg) {
            n = aiger_strip(n);
        }
        AigerMap::iterator it = cache.find(n);
        assert(it != cache.end());
        Aig next = it->second;
        if (neg) {
            next = mgr_.aig_not(next);
        }

        n = aigmgr->latches[i].reset;
        neg = aiger_sign(n);
        if (neg) {
            n = aiger_strip(n);
        }
        it = cache.find(n);
        assert(it != cache.end());
        Aig init = it->second;
        if (neg) {
            init = mgr_.aig_not(init);
        }            

        Aig lv = mgr_.aig_var(aiger_lit2var(lit));
        add_aiger_latch(lv, init, next);
    }

    for (unsigned int i = 0; i < aigmgr->num_outputs; ++i) {
        unsigned int lit = aigmgr->outputs[i].lit;

        bool neg = aiger_sign(lit);
        if (neg) {
            lit = aiger_strip(lit);
        }
        AigerMap::iterator it = cache.find(lit);
        assert(it != cache.end());
        Aig p = it->second;
        if (!neg) {
            p = mgr_.aig_not(p);
        }
        add_aiger_output(p);
    }

    for (unsigned int i = 0; i < aigmgr->num_bad; ++i) {
        unsigned int lit = aigmgr->bad[i].lit;

        bool neg = aiger_sign(lit);
        if (neg) {
            lit = aiger_strip(lit);
        }
        AigerMap::iterator it = cache.find(lit);
        assert(it != cache.end());
        Aig p = it->second;
        if (!neg) {
            p = mgr_.aig_not(p);
        }
        add_aiger_output(p);
    }
    
    return true;
}


bool Model::init_from_aiger(const std::string &filename)
{
    aiger *aigmgr = aiger_init();
    const char *error = aiger_open_and_read_from_file(aigmgr, filename.c_str());
    bool ret = (error == NULL);

    if (ret) {
        ret = init_from_aiger(aigmgr);
    }

    aiger_reset(aigmgr);
    
    return ret;
}


} // namespace simplic3
