// -*- C++ -*-
// 
// SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
// for finite-state functional transition systems
//
// Simulation support
//
// Author: Alberto Griggio <griggio@fbk.eu>
// See LICENSE.txt for copyright/licensing information
// See CREDITS.txt for other credits
//

#include "modelsim.h"
#include "miscutils.h"

namespace simplic3 {

ModelSim::ModelSim(Model *model):
    model_(model)
{
}


ModelSim::~ModelSim()
{
}


void ModelSim::init(const std::vector<value> &initial_state)
{
    current_.clear();
    cache_.clear();
    cache_[model_->aig_manager()->aig_true()] = TRUE;

    for (size_t i = 0; i < initial_state.size(); ++i) {
        Aig a = model_->statevars()[i];
        cache_[a] = initial_state[i];
        current_.push_back(initial_state[i]);
    }
}


ModelSim::value ModelSim::mkval(value v, bool neg)
{
    if (!neg) {
        return v;
    }
    
    switch (v) {
    case FALSE: return TRUE;
    case TRUE: return FALSE;
    default: return UNDEF;
    }
}


ModelSim::value ModelSim::mkand(value l, value r)
{
    if (l == FALSE || r == FALSE) {
        return FALSE;
    }
    if (l == TRUE && r == TRUE) {
        return TRUE;
    }
    return UNDEF;
}


void ModelSim::step(const std::vector<value> &inputs)
{
    cache_.clear();
    cache_[model_->aig_manager()->aig_true()] = TRUE;
    
    for (size_t i = 0; i < current_.size(); ++i) {
        Aig a = model_->statevars()[i];
        cache_[a] = current_[i];
    }
    
    for (size_t i = 0; i < inputs.size(); ++i) {
        Aig a = model_->inputs()[i];
        cache_[a] = inputs[i];
    }

    // evaluate properties
    for (AigList::const_iterator it = model_->properties().begin(),
             end = model_->properties().end(); it != end; ++it) {
        Aig a = *it;
        get(a);
    }

    // compute next state
    for (size_t i = 0; i < current_.size(); ++i) {
        Aig a = model_->statevars()[i];
        Aig n = model_->next_value(a);
        current_[i] = get(n);
    }

    for (size_t i = 0; i < current_.size(); ++i) {
        Aig a = model_->statevars()[i];
        cache_[a] = current_[i];
    }
    
}


ModelSim::value ModelSim::get(Aig a)
{
    to_process_.clear();
    to_process_.push_back(a);

    while (!to_process_.empty()) {
        Aig cur = to_process_.back();
        Aig v = aig_var(cur);
        if (cache_.find(v) != cache_.end()) {
            to_process_.pop_back();
            continue;
        }

        if (AigManager::aig_is_and(v)) {
            Aig l = AigManager::aig_get_left(v);
            Aig r = AigManager::aig_get_right(v);

            Aig lv = aig_var(l);
            Aig rv = aig_var(r);

            bool children_done = true;
            if (cache_.find(rv) == cache_.end()) {
                children_done = false;
                to_process_.push_back(rv);
            }
            if (cache_.find(lv) == cache_.end()) {
                children_done = false;
                to_process_.push_back(lv);
            }

            if (children_done) {
                to_process_.pop_back();

                value vl = mkval(cache_[lv], aig_sign(l));
                value vr = mkval(cache_[rv], aig_sign(r));

                value val = mkand(vl, vr);
                cache_[v] = val;
            }
        } else {
            cache_[v] = UNDEF;
            to_process_.pop_back();
        }
    }

    value ret = cache_[aig_var(a)];
    ret = mkval(ret, aig_sign(a));
    return ret;
}


} // namespace simplic3
