#!/usr/bin/env python 

# SimplIC3: a "simple" implementation of IC3 (and other SAT-based algorithms)
# for finite-state functional transition systems
# 
# Distutils setup file for building the Python API
# 
# Author: Alberto Griggio <griggio@fbk.eu>
# See LICENSE.txt for copyright/licensing information
# See CREDITS.txt for other credits

import os, sys
import optparse
import subprocess
from distutils.core import setup, Extension


def parse_cmake_cache(pth):
    res = {}
    with open(pth) as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith('#') or line.startswith('//'):
                continue
            bits = line.split('=', 1)
            if len(bits) == 2:
                val = bits[1]
                key = bits[0].split(':', 1)[0]
                res[key] = val
    return res


def getbool(val):
    return {'1': True, 'true': True, 'on': True,
            '0': False, 'false': False, 'off': False}[val.lower()]


def main():
    p = optparse.OptionParser('%prog [options] [-- [setup_options]]')
    p.add_option('--simplic3-build-dir', help='Build dir of simplic3')
    p.add_option('--extra-compile-args', help='extra compilation args',
                 default="")
    p.add_option('--extra-link-args', help='extra link args', default="")
    p.add_option('--swig-only', help='only run SWIG '
                 '(do not compile generated bindings)', action='store_true')
    p.add_option('--no-swig', help='do not run SWIG', action='store_true')

    try:
        idx = sys.argv.index('--')
        optargs = sys.argv[1:idx]
        argv = sys.argv[idx+1:]
    except ValueError:
        optargs, argv = sys.argv[1:], []
    argv.append(sys.argv[0])
    sys.argv = argv

    def error(msg):
        sys.stderr.write(msg)
        sys.stderr.write('\n')
        p.print_help(sys.stderr)
        sys.exit(-1)

    opts, args = p.parse_args(optargs)

    if args:
        sys.argv = [sys.argv[0]] + args + sys.argv[1:]

    if not opts.simplic3_build_dir:
        error("--simplic3-build-dir is required")
    else:
        LIBSIMPLIC3_A_DIR = opts.simplic3_build_dir

    cmake_cache = parse_cmake_cache(os.path.join(opts.simplic3_build_dir,
                                                 'CMakeCache.txt'))
    SIMPLIC3_H_DIR = cmake_cache['simplic3_SOURCE_DIR']

    extra_libs = []
    if getbool(cmake_cache['ENABLE_MINISAT']):
        extra_libs.append(cmake_cache['_minisat_lib'])

    if getbool(cmake_cache['ENABLE_PICOSAT']):
        extra_libs.append(cmake_cache['_picosat_lib'])

    if not opts.no_swig:
        print "Generating Python wrapper with SWIG..."
        swigfile = os.path.join(os.path.dirname(sys.argv[0]),
                                'simplic3_python.i')
        extra_includes = [n for n in opts.extra_compile_args.split()
                          if n.startswith('-I')]
        p = subprocess.Popen(['swig', '-I%s' % SIMPLIC3_H_DIR, '-python',
                              '-o', 'simplic3_python_wrap.c', swigfile])
        p.communicate()
        swig_ok = p.returncode == 0
    else:
        swig_ok = True

    if swig_ok and not opts.swig_only:
        print "Compiling the extension module..."
        extra_compile_args = ['-DSWIG', '-g',
                              '-Wno-unused',
                              '-Wno-uninitialized',
                              '-I' + SIMPLIC3_H_DIR] + \
                              opts.extra_compile_args.split()
        extra_link_args = ['-g', '-L' + LIBSIMPLIC3_A_DIR] + \
                          opts.extra_link_args.split() + extra_libs
        libraries = ['simplic3', 'stdc++']
        setup(name='simplic3', version='0.1',
              description='SimplIC3 API',
              ext_modules=[Extension('_simplic3', ['simplic3_python_wrap.c'],
                                     undef_macros=['NDEBUG'],
                                     extra_compile_args=extra_compile_args,
                                     extra_link_args=extra_link_args,
                                     libraries=libraries,
                                     )]
              )


if __name__ == '__main__':
    main()
